/*
 * Copyright 2026 ej-technologies GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 */
package com.jprofiler.api.controller;

import java.io.File;

/**
 * <p>
 * Use this class to influence the run time behaviour of JProfiler.
 * </p>
 * <b>Overhead</b>
 * <p>
 * When executing your application without JProfiler, all calls into this class will quietly do nothing without any
 * overhead, because the implementations of the methods are empty. When the JProfiler agent is present, these methods will
 * be instrumented and wired up with the agent.
 * </p>
 * <b>Interactive use</b>
 * <p>If you would like to access the functionality in this class from the command line, use the
 * {@code jpcontroller} command line executable that attaches to the JProfiler MBean which is registered
 * for every profiled application. For more information, invoke {@code jpcontroller -help}.
 * </p>
 * <b>Offline profiling</b>
 * <p>
 * While you can use the controller during interactive profiling sessions, it is most useful for <b>offline profiling</b>.
 * "Offline" refers to the fact that the JProfiler GUI does not have to be connected to the profiled process while
 * profiling data is being recorded and snapshots are saved.
 * </p>
 * <p>
 * Please read the "Offline profiling" chapter in the documentation (directory {@code $JPROFILER_HOME/doc}) for an
 * overview on how to use the Controller API and consult the javadoc for {@link com.jprofiler.api.controller.Controller}.
 * </p>
 * <p>
 * The {@code $JPROFILER_HOME/api/samples/offline} example project shows how to perform offline profiling.
 * Open the gradle build file {@code build.gradle} in an editor for instructions on how to run it.
 * </p>
 * <b>JAR file and dependency</b>
 * <p>
 * All classes in this package are contained in {@code $JPROFILER_HOME/api/jprofiler-controller.jar}.
 * The controller API is also available as a dependency. In the snippets below, "VERSION"
 * should be replaced with the corresponding JProfiler version.
 * </p>
 * <b>Maven</b>
 * <pre>{@code
 * <dependency>
 *   <groupId>com.jprofiler</groupId>
 *   <artifactId>jprofiler-controller</artifactId>
 *   <version>VERSION</version>
 * </dependency>
 * }</pre>
 * <b>Gradle</b>
 * <pre>{@code
 * dependencies {
 *   classpath group: 'com.jprofiler', name: 'jprofiler-controller', version: 'VERSION'
 * }
 * }</pre>
 *
 */
@SuppressWarnings("unused")
public class Controller {
    private static final String PROBE_NAME_PREFIX_BUILTIN = "builtin.";
    /**
     * Name of the built-in persistence probe. Used in {@link #startProbeRecording(String, boolean)} and {@link #stopProbeRecording(String)}.
     */
    public static final String PROBE_NAME_PERSISTENCE = PROBE_NAME_PREFIX_BUILTIN + "PersistenceProbe";

    /**
     * Name of the built-in JDBC probe. Used in {@link #startProbeRecording(String, boolean)} and {@link #stopProbeRecording(String)}.
     */
    public static final String PROBE_NAME_JDBC = PROBE_NAME_PREFIX_BUILTIN + "JdbcProbe";

    /**
     * Name of the built-in JMS probe. Used in {@link #startProbeRecording(String, boolean)} and {@link #stopProbeRecording(String)}.
     */
    public static final String PROBE_NAME_JMS = PROBE_NAME_PREFIX_BUILTIN + "JmsProbe";

    /**
     * Name of the built-in JNDI probe. Used in {@link #startProbeRecording(String, boolean)} and {@link #stopProbeRecording(String)}.
     */
    public static final String PROBE_NAME_JNDI = PROBE_NAME_PREFIX_BUILTIN + "JndiProbe";

    /**
     * Name of the built-in MongoDB probe. Used in {@link #startProbeRecording(String, boolean)} and {@link #stopProbeRecording(String)}.
     */
    public static final String PROBE_NAME_MONGO_DB = PROBE_NAME_PREFIX_BUILTIN + "MongoDbProbe";

    /**
     * Name of the built-in Cassandra probe. Used in {@link #startProbeRecording(String, boolean)} and {@link #stopProbeRecording(String)}.
     */
    public static final String PROBE_NAME_CASSANDRA = PROBE_NAME_PREFIX_BUILTIN + "CassandraProbe";

    /**
     * Name of the built-in HTTP client probe. Used in {@link #startProbeRecording(String, boolean)} and {@link #stopProbeRecording(String)}.
     */
    public static final String PROBE_NAME_HTTP_CLIENT = PROBE_NAME_PREFIX_BUILTIN + "HttpClientProbe";

    /**
     * Name of the built-in RMI probe. Used in {@link #startProbeRecording(String, boolean)} and {@link #stopProbeRecording(String)}.
     */
    public static final String PROBE_NAME_RMI = PROBE_NAME_PREFIX_BUILTIN + "RmiProbe";

    /**
     * Name of the built-in gRPC probe. Used in {@link #startProbeRecording(String, boolean)} and {@link #stopProbeRecording(String)}.
     */
    public static final String PROBE_NAME_GRPC = PROBE_NAME_PREFIX_BUILTIN + "GrpcProbe";

    /**
     * Name of the built-in AI probe. Used in {@link #startProbeRecording(String, boolean)} and {@link #stopProbeRecording(String)}.
     */
    public static final String PROBE_NAME_AI = PROBE_NAME_PREFIX_BUILTIN + "AiProbe";

    /**
     * Name of the built-in Apache Kafka producer probe. Used in {@link #startProbeRecording(String, boolean)} and {@link #stopProbeRecording(String)}.
     */
    public static final String PROBE_NAME_KAFKA_PRODUCER = PROBE_NAME_PREFIX_BUILTIN + "KafkaProducerProbe";

    /**
     * Name of the built-in Apache Kafka consumer probe. Used in {@link #startProbeRecording(String, boolean)} and {@link #stopProbeRecording(String)}.
     */
    public static final String PROBE_NAME_KAFKA_CONSUMER = PROBE_NAME_PREFIX_BUILTIN + "KafkaConsumerProbe";

    /**
     * Name of the built-in web service probe. Used in {@link #startProbeRecording(String, boolean)} and {@link #stopProbeRecording(String)}.
     */
    public static final String PROBE_NAME_WS = PROBE_NAME_PREFIX_BUILTIN + "WsProbe";

    /**
     * Name of the built-in http server probe. Used in {@link #startProbeRecording(String, boolean)} and {@link #stopProbeRecording(String)}.
     */
    public static final String PROBE_NAME_HTTP_SERVER = PROBE_NAME_PREFIX_BUILTIN + "HttpServerProbe";

    /**
     * Name of the built-in file probe. Used in {@link #startProbeRecording(String, boolean)} and {@link #stopProbeRecording(String)}.
     */
    public static final String PROBE_NAME_FILE = PROBE_NAME_PREFIX_BUILTIN + "FileProbe";

    /**
     * Name of the built-in socket probe. Used in {@link #startProbeRecording(String, boolean)} and {@link #stopProbeRecording(String)}.
     */
    public static final String PROBE_NAME_SOCKET = PROBE_NAME_PREFIX_BUILTIN + "SocketProbe";

    /**
     * Name of the built-in process probe. Used in {@link #startProbeRecording(String, boolean)} and {@link #stopProbeRecording(String)}.
     */
    public static final String PROBE_NAME_PROCESS = PROBE_NAME_PREFIX_BUILTIN + "ProcessProbe";

    /**
     * Name of the built-in class loader probe. Used in {@link #startProbeRecording(String, boolean)} and {@link #stopProbeRecording(String)}.
     */
    public static final String PROBE_NAME_CLASS_LOADER = PROBE_NAME_PREFIX_BUILTIN + "ClassLoaderProbe";

    /**
     * Name of the built-in Exception probe. Used in {@link #startProbeRecording(String, boolean)} and {@link #stopProbeRecording(String)}.
     */
    public static final String PROBE_NAME_EXCEPTION = PROBE_NAME_PREFIX_BUILTIN + "ExceptionProbe";

    /**
     * Name of the built-in GC probe. Used in {@link #startProbeRecording(String, boolean)} and {@link #stopProbeRecording(String)}.
     */
    public static final String PROBE_NAME_GC = PROBE_NAME_PREFIX_BUILTIN + "GCProbe";

    /**
     * Identifier for all custom probes. If used in {@link #startProbeRecording(String, boolean)} and {@link #stopProbeRecording(String)} all
     * custom probes will be started or stopped.
     */
    public static final String PROBE_NAME_ALL_CUSTOM = PROBE_NAME_PREFIX_BUILTIN + "AllCustom";

    /**
     * Starts recording CPU data.
     * This method can be called repeatedly and alternatingly with
     * {@code stopCPURecording()}. With these methods you can restrict
     * CPU profiling to certain regions of your code.
     *
     * @param reset if {@code true} and cpu recording is currently switched off,
     *              any previously accumulated CPU profiling data will be discarded.
     *              If {@code false}, CPU data will be accumulated across pairs of
     *              invocations of {@code startCPURecording()} and {@code stopCPURecording()}.
     */
    public static void startCPURecording(boolean reset) {
    }

    /**
     * Starts recording CPU data.
     * This method can be called repeatedly and alternatingly with
     * {@code stopCPURecording()}. With these methods you can restrict
     * CPU profiling to certain regions of your code.
     *
     * @param reset if {@code true} and cpu recording is currently switched off,
     *              any previously accumulated CPU profiling data will be discarded.
     *              If {@code false}, CPU data will be accumulated across pairs of
     *              invocations of {@code startCPURecording()} and {@code stopCPURecording()}.
     * @param trackingOptions an array of TRACKING_ constants or {@code null} for the previous state
     */
    public static void startCPURecording(boolean reset, TrackingOptions trackingOptions) {
    }

    /**
     * Stops CPU recording.
     * This method can be called after {@code startCPURecording()} has been called.
     * However, you do not have to call it since CPU profiling can run until the
     * JVM exits.
     */
    public static void stopCPURecording() {
    }

    /**
     * Start recording data for a selected probe. For more information and a more general signature see
     * {@link #startProbeRecording(String, ProbeRecordingOptions)}.
     */
    public static void startProbeRecording(String probeName, boolean events) {
    }

    /**
     * Start recording data for a selected probe.
     * @param probeName the name of the probe. For built-in probes, see the {@code PROBE_NAME} constants in this class.
     * For embedded and injected probes, the class name of the probe is used.
     * For script probes, this name is {@code script.n} where n is the one-based index of the script probe.
     * @param recordingOptions determines whether special recordings for the probe should be switched on
     * @see #stopProbeRecording(String)
     * @see ProbeRecordingOptions
     */
    public static void startProbeRecording(String probeName, ProbeRecordingOptions recordingOptions) {
    }

    /**
     * Stop recording data for a selected probe.
     * @param probeName the name of the probe. For built-in probes, see the {@code PROBE_NAME} constants in this class.
     * For embedded and injected probes, the class name of the probe is used.
     * For script probes, this name is {@code script.n} where n is the one-based index of the script probe.
     * @see #startProbeRecording(String, boolean)
     */
    public static void stopProbeRecording(String probeName) {
    }

    /**
     * Start tracking selected elements for a selected probe. Probe recording must be switched on for the selected
     * probe otherwise this method does not have any effect.
     * @param probeName the name of the probe. For built-in probes, see the {@code PROBE_NAME} constants in this class.
     * For embedded and injected probes, the class name of the probe is used.
     * For script probes, this name is {@code script.n} where n is the one-based index of the script probe.
     * @param descriptions the descriptions of the tracked elements. These are the strings that you see in the probe view in the JProfiler GUI.
     *                     The sum of the selected elements will be tracked in a single graph. To get separate graphs,
     *                     call this method multiple times.
     * @param objectType the type of the elements to be tracked
     * @param valueType the type of the values to be tracked
     */
    public static void startProbeTracking(String probeName, String[] descriptions, ProbeObjectType objectType, ProbeValueType valueType) {
    }

    /**
     * Stop tracking selected elements for a selected probe. This only has an effect if
     * {@link #startProbeTracking(String, String[], ProbeObjectType, ProbeValueType)} has been called before.
     * @param probeName the name of the probe. For built-in probes, see the {@code PROBE_NAME} constants in this class.
     * For embedded and injected probes, the class name of the probe is used.
     * For script probes, this name is {@code script.n} where n is the one-based index of the script probe.
     * @param description the descriptions of the tracked elements
     * @param objectType the type of the elements to be tracked
     * @param valueType the type of the values to be tracked
     */
    public static void stopProbeTracking(String probeName, String[] description, ProbeObjectType objectType, ProbeValueType valueType) {
    }

    /**
     * Starts algorithmic complexity recording for all previously configured methods.
     */
    public static void startComplexityRecording() {
    }

    /**
     * Stops algorithmic complexity recording and prepares data for snapshot.
     */
    public static void stopComplexityRecording() {
    }

    /**
     * Starts the call tracer.
     * This method can be called repeatedly and alternatingly with
     * {@code stopCallTracer()}.
     *
     * @param recordFiltered if {@code true}, calls into filtered classes will be recorded, too.
     * @param cap the maximum number of events to be recorded. A good default is 100000.
     * @param reset if {@code true} previously recorded calls will be cleared..
     */
    public static void startCallTracer(int cap, boolean recordFiltered, boolean reset) {
    }

    /**
     * Stops the call tracer.
     * This method can be called after {@code startCallTracer()} has been called.
     * However, you do not have to call it since the call tracer will stop automatically
     * after the cap has been reached. The data will be delivered to the frontend afterwards
     * if used in online mode.
     */
    public static void stopCallTracer() {
    }

    /**
     * Start recording of monitor usage with default thresholds of 100 microseconds
     * for blocking events and 100 ms for waiting events.
     * This method can be called repeatedly and alternatingly with
     * {@code stopMonitorRecording()}. Monitor profiling is switched off
     * by default.
     */
    public static void startMonitorRecording() {
    }

    /**
     * Start recording of monitor usage.
     * This method can be called repeatedly and alternatingly with
     * {@code stopMonitorRecording()}. Monitor profiling is switched off
     * by default.
     * @param blockingThreshold the recording threshold for blocking events in microseconds.
     * @param waitingThreshold the recording threshold for waiting events in microseconds.
     */
    public static void startMonitorRecording(int blockingThreshold, int waitingThreshold) {
    }

    /**
     * Stop recording of monitor usage.
     * This method can be called repeatedly and alternatingly with
     * {@code startMonitorRecording()}. However, you do not have to call it since
     * monitor profiling can run until the JVM exits.
     */
    public static void stopMonitorRecording() {
    }

    /**
     * Starts recording of memory allocations.
     * This method can be called repeatedly and alternatingly with
     * {@code stopAllocRecording()}. With these methods you can restrict
     * memory allocation profiling to certain regions of your code.
     * This is especially useful for profiling an application running within
     * an application server.
     *
     * @param reset if {@code true} and allocation recording is currently switched off,
     *              any previously recorded profiling data will be discarded.
     *              If {@code false}, allocations within all pairs of invocations of
     *              {@code startAllocRecording()} and {@code stopAllocRecording()} will be recorded.
     */
    public static void startAllocRecording(boolean reset) {
    }

    /**
     * Starts recording of memory allocations.
     * This method can be called repeatedly and alternatingly with
     * {@code stopAllocRecording()}. With these methods you can restrict
     * memory allocation profiling to certain regions of your code.
     * This is especially useful for profiling an application running within
     * an application server.
     *
     * @param reset if {@code true} and allocation recording is currently switched off,
     *              any previously recorded profiling data will be discarded.
     *              If {@code false}, allocations within all pairs of invocations of
     *              {@code startAllocRecording()} and {@code stopAllocRecording()} will be recorded.
     * @param trackingOptions the request tracking options
     */
    public static void startAllocRecording(boolean reset, TrackingOptions trackingOptions) {
    }


    /**
     * Stops recording of memory allocations.
     * This method can be called after {@code startAllocRecording()} has been called.
     * However, you do not have to call it since memory profiling can run until the
     * JVM exits.
     */
    public static void stopAllocRecording() {
    }

    /**
     * Adds a bookmark at the current time.
     * The bookmark will be displayed in all JProfiler graphs with a time axis.
     * The description will be displayed in the tooltip for the bookmark.
     *
     * @param description the name of the bookmark, may also be {@code null}
     */
    public static void addBookmark(String description) {
    }

    /**
     * Add a bookmark at the current time.
     * The bookmark will be displayed in all JProfiler graphs with a time axis.
     * The description will be displayed in the tooltip for the bookmark.
     * @param description the name of the bookmark, may also be {@code null}
     * @param color the RGB color to be used for drawing the bookmark, for example {@code 0xFF0000}.
     * @param dashed if the line for drawing the bookmark should be dashed or not
     */
    public static void addBookmark(String description, int color, boolean dashed) {
    }

    /**
     * Trigger a thread dump.
     *
     * @param includeVirtualThreads if {@code true}, virtual threads will be included. The thread dump won't be synchronized in this case.
     */
    public static void triggerThreadDump(boolean includeVirtualThreads) {
    }

    /**
     * Trigger a thread dump with an optional label for identification.
     *
     * @param includeVirtualThreads if {@code true}, virtual threads will be included. The thread dump won't be synchronized in this case.
     * @param label A string parameter used to label the dump. Can be null.
     */
    public static void triggerThreadDump(boolean includeVirtualThreads, String label) {
    }

    /**
     * Triggers a dump of all objects on the heap to be displayed it in the "All objects" view.
     */
    public static void triggerAllObjectsDump() {
    }

    /**
     * Triggers a dump of all objects on the heap to be displayed it in the "All objects" view
     * with an optional label for identification.
     *
     * @param label A string parameter used to label the dump. Can be null.
     */
    public static void triggerAllObjectsDump(String label) {
    }


    /**
     * Trigger a dump of current monitors and locks.
     */
    public static void triggerMonitorDump() {
    }

    /**
     * Trigger a heap dump.
     * Calls {@code triggerHeapDump(HeapDumpOptions.DEFAULT)}.
     * @see #triggerHeapDump(HeapDumpOptions)
     */
    public static void triggerHeapDump() {
    }

    /**
     * Trigger a heap dump with the specified {@link HeapDumpOptions heap dump options}.
     * If you want to analyze a heap dump with the heap walker in a snapshot saved with
     * the {@code saveSnapshot} method, you should call this method from your source code
     * at an appropriate time.
     * <p>
     * <b>ATTENTION:</b> Taking a heap dump takes a long time (on the order of seconds).
     * If you call this method to often, your application might become unusable or take
     * an excessively long time to finish.
     * </p>
     * @param heapDumpOptions the options for the heap dump
     */
    public static void triggerHeapDump(HeapDumpOptions heapDumpOptions) {
    }

    /**
     * Mark all objects currently on the heap as "old". In the next heap dump, you can then distinguish between objects created before and after this method was called.
     * This action has much less overhead than a full heap dump
     */
    public static void markHeap() {
    }

    /**
     * Saves a snapshot of all profiling data to disk.
     * This is especially important for offline profiling.
     * You should choose the standard extension {@code .jps} for the
     * {@code file} parameter, since JProfiler's GUI frontend filters the corresponding
     * file choosers for that extension. If you want to save several snapshots during
     * one profiling run, take care to provide unique {@code file} parameters
     * since snapshot files will be overwritten otherwise.
     * <p>
     * <b>ATTENTION:</b> Saving a snapshot takes a long time (on the order of seconds).
     * If you call this method to often, your application might become unusable or take
     * an excessively long time to finish, and your hard disk might run out of space.
     * </p>
     *
     * @param file the file to which the snapshot should be saved.
     */
    public static void saveSnapshot(final File file) {
    }

    /**
     * Starts recording of thread states and monitor usage.
     * This method can be called repeatedly and alternatingly with
     * {@code stopThreadProfiling()}. For an offline session, thread
     * profiling is switched on by default.
     */
    public static void startThreadProfiling() {
    }

    /**
     * Stops recording of thread states and monitor usage.
     * This method can be called repeatedly and alternatingly with
     * {@code startThreadProfiling()}. However, you do not have to call it since
     * thread profiling can run until the JVM exits.
     */
    public static void stopThreadProfiling() {
    }

    /**
     * Starts recording of VM telemetry data.
     * This method can be called repeatedly and alternatingly with
     * {@code stopVMTelemetryRecording()}. For an offline session, VM telemetry
     * recording is switched on by default.
     */
    public static void startVMTelemetryRecording() {
    }

    /**
     * Stops recording of VM telemetry data.
     * This method can be called repeatedly and alternatingly with
     * {@code startVMTelemetryRecording()}. However, you do not have to call it since
     * VM telemetry recording can run until the JVM exits.
     */
    public static void stopVMTelemetryRecording() {
    }

    /**
     * Saves a snapshot of all profiling data to disk when the VM shuts down.
     * This is especially important for offline profiling.
     * You should choose the standard extension {@code .jps} for the
     * {@code file} parameter, since JProfiler's GUI frontend filters the corresponding
     * file choosers for that extension.
     * <p>
     * <b>ATTENTION:</b> Saving a snapshot can take quite some time (on the order of seconds).
     * When the VM is shut down during a user logout or a system shutdown, the OS may terminate
     * the VM before saving is completed.
     *
     * @param file the file to which the snapshot should be saved.
     */
    public static void saveSnapshotOnExit(final File file) {
    }

    /**
     * Enable or disable all triggers with a specified group ID. The group ID can be entered in
     * the "Group ID" step of the trigger configuration wizard in the JProfiler GUI.
     * @param enabled if the triggers should be enabled
     * @param groupId the group ID
     * @throws IllegalArgumentException if no trigger with the specified group ID exists
     */
    public static void enableTriggerGroup(boolean enabled, String groupId) throws IllegalArgumentException {
    }

    /**
     * Enable or disable all triggers. The enabled/disabled state of the single triggers will not be lost,
     * disabling all triggers with this method overrides the enabled/disabled state of the single triggers.
     * @param enabled if the triggers should be enabled
     */
    public static void enableTriggers(boolean enabled) {
    }

    /**
     * Triggers an MBean snapshot with an optional label. The recorded MBeans con be
     * optionally filtered with a regular expression.
     *
     * @param label An optional label for the snapshot. Can be null.
     * @param regexp An optional regular expression for filtering MBeans based on the canonical name. Can be null.
     */
    public static void triggerMBeanSnapshot(String label, String regexp) {
    }

    private Controller() {
    }
}
