package com.jprofiler.api.agent;

/**
 * Represents a proxy to a Kafka {@code ProducerRecord} combined with selected data from {@code RecordMetadata}.
 * Provides access to details about the topic, partition, key, value, and metadata related to the Kafka record.
 * Acts as an abstraction layer to work with Kafka ProducerRecord objects and their associated metadata.
 */
public interface KafkaProducerRecord {

    /**
     * Gets the topic to which the record is sent.
     *
     * @return the name of the Kafka topic.
     */
    String topic();

    /**
     * Gets the partition to which the record was sent from the records' metadata.
     *
     * @return the partition ID
     */
    int partition();

    /**
     * Gets the value of the last header with the specified key from the record’s headers.
     *
     * @param key the key of the desired header.
     * @return the value of the last header with the given key as a byte array,
     * or {@code null} if the header does not exist.
     */
    byte[] lastHeaderValue(String key);

    /**
     * Gets the deserialized key of the record.
     *
     * @return the key of the record, or {@code null} if no key is present.
     */
    Object key();

    /**
     * Gets the deserialized value of the record.
     *
     * @return the value of the record, or {@code null} if no value is present.
     */
    Object value();

    /**
     * Gets the size of the serialized key, in bytes.
     *
     * @return the size of the serialized key, or {@code -1} if the key is not present.
     */
    int serializedKeySize();

    /**
     * Gets the size of the serialized value, in bytes.
     *
     * @return the size of the serialized value, or {@code -1} if the value is not present.
     */
    int serializedValueSize();

    /**
     * Gets the underlying producer record represented by this proxy.
     * This method returns the original Kafka {@code ProducerRecord} object.
     *
     * @return the original producer record
     */
    Object producerRecord();

    /**
     * Gets the underlying metadata represented by this proxy.
     * This method returns the original Kafka {@code RecordMetadata} object.
     *
     * @return the original metadata
     */
    Object metadata();
}