/*
 * Copyright 2025 ej-technologies GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 */
package com.jprofiler.api.probe.embedded;

import java.util.concurrent.Callable;

/**
 * By calling this class you can add an arbitrary split in the call tree, just like the URL splitting of JProfiler's built-in "HTTP Server" probe.
 * <p>You can either use a pair of {@link #enter(Class, String)} and {@link #exit()} calls or the two {@code execute} wrappers. If you write your own
 * wrapper in another JVM language it is advisable to use the {@link #enter(Class, String)} and {@link #exit()} primitives.</p>
 * <p>When using the primitives, it is very important that a matching {@link #exit()} is always called, so the exit call must be put it a finally block like this:</p>
 * <pre>{@code
 *     Split.enter(MyProbe.class, "my split string");
 *     try {
 *         // your code
 *     } finally {
 *         Split.exit();
 *     }
 * }</pre>
 *
 */
@JProfilerEmbeddedClass
public class Split {
    /**
     * Split the call tree with the given {@code name}. A matching {@link #exit()} call must be always called.
     * 
     * @param probe the configuration class. Must not be {@code null}.
     * @param name the line that should show up in the call tree. If {@code null}, the call tree won't be split.
     */
    public static void enter(Class<? extends SplitProbe> probe, String name) {
    }

    /**
     * Marks the end of a split. A matching {@link #enter(Class, String)} call must have been called first.
     */
    public static void exit() {
    }

    /**
     * A wrapper for the {@link #enter(Class, String)} and {@link #exit()} primitives for a {@link Runnable}.
     * @param probe the configuration class. Must not be {@code null}.
     * @param name the line that should show up in the call tree. If {@code null}, the call tree won't be split.
     * @param runnable the runnable that will be executed inside the split.
     */
    public static void execute(Class<? extends SplitProbe> probe, String name, Runnable runnable) {
        enter(probe, name);
        try {
            runnable.run();
        } finally {
            exit();
        }
    }

    /**
     * A wrapper for the {@link #enter(Class, String)} and {@link #exit()} primitives for a {@link Callable}.
     * @param probe the configuration class. Must not be {@code null}.
     * @param name the line that should show up in the call tree. If {@code null}, the call tree won't be split.
     * @param callable the callable that will be executed inside the split.
     */
    public static <T> T execute(Class<? extends SplitProbe> probe, String name, Callable<T> callable) throws Exception {
        enter(probe, name);
        try {
            return callable.call();
        } finally {
            exit();
        }
    }

    /**
     * You can register your probe independently of {@link #enter(Class, String)} and {@code execute} calls. As soon as the probe is registered
     * it will be displayed in the JProfiler UI and telemetries can be recorded.
     * You don't have to call this method if you don't want to start telemetry recording before an {@link #enter(Class, String)} or {@code execute} call
     * takes place. {@link #enter(Class, String)} or {@code execute} will automatically register your probe.
     *
     * @param probe the configuration class. Must not be {@code null}.
     */
    public static void register(Class<? extends SplitProbe> probe) {
    }

    static {
        JProfilerEmbeddedPackageMarker.register();
    }
}
