/*
 * Copyright 2025 ej-technologies GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 */
package com.jprofiler.api.probe.embedded;

import java.util.concurrent.Callable;

/**
 * By calling this class you can add entries to the payload call tree and hotspots of your probe.
 * <p>You can either use a pair of {@link #enter(Class)} and {@link #exit(String)} calls or the {@code execute} wrappers. If you write your own
 * wrapper in another JVM language it is advisable to use the {@link #enter(Class)} and {@link #exit(String)} primitives.</p>
 * <p>When using the primitives, it is very important that a matching {@link #exit(String)} is always called, so the exit call must be put it a finally block like this:</p>
 * <pre>{@code
 *     Payload.enter(MyProbe.class);
 *     try {
 *         // your code
 *     } finally {
 *         Payload.exit("my payload string");
 *     }
 * }</pre>
 *<p>If you use control objects in your probe (for example to record connections like in the JDBC probe) you can use the {@link #enter(Class, Object, Enum)} or the similarly overloaded {@code execute} wrappers
 * to associate a payload with a control object. With the {@link #openControlObject(Class, Object, String)}, {@link #closeControlObject(Class, Object)} and
 * {@link PayloadProbe#getControlObjectName(Object)} methods you can further customize control object recording. Your configuration subclass of {@link PayloadProbe} must override
 * {@link PayloadProbe#isControlObjects()} and return {@code true} to enable control object recording as a capability for the probe.</p>
 * <p>If you want to record different event types which also correspond to different control object states, you have to override {@link PayloadProbe#getCustomTypes()} and return a custom enum class.
 * Objects of this class can then be passed to {@link #enter(Class, Object, Enum)}.
 * The {@code toString()} method of the enums will be used to get the event name. If you want to control the event color, the enum class can implement the {@link TypeCustomizer} interface.</p>
 */
@JProfilerEmbeddedClass
public class Payload {
    /**
     * Start a payload. A matching {@link #exit(String)} call must be always called.
     *
     * @param probe the configuration class. Must not be {@code null}.
     */
    public static void enter(Class<? extends PayloadProbe> probe) {
    }

    /**
     * Start a payload. A matching {@link #exit(String)} call must be always called.
     *  @param probe the configuration class. Must not be {@code null}.
     * @param controlObject the control object this payload belongs to or {@code null}. If control objects should be recorded you have to override {@link PayloadProbe#isControlObjects()}.
     * @param type the payload type. Must by an object of your custom enum class registered by overriding {@link PayloadProbe#getCustomTypes()} or {@code null}.
     */
    public static void enter(Class<? extends PayloadProbe> probe, Object controlObject, Enum<?> type) {
    }

    /**
     * Finishes recording of a payload. A matching {@code enter} call must have been called first.
     *
     * @param name the name of the recorded payload. If {@code null}, the payload recording will be discarded.
     */
    public static void exit(String name) {
    }

    /**
     * A wrapper for the {@link #enter(Class)} and {@link #exit(String)} primitives for a {@link Runnable}.

     * @param probe the configuration class. Must not be {@code null}.
     * @param name the name of the recorded payload. If {@code null}, the payload recording will be discarded.
     * @param runnable the runnable that will be timed for the payload recording.
     */
    public static void execute(Class<? extends PayloadProbe> probe, String name, Runnable runnable) {
        execute(probe, name, null, null, runnable);
    }

    /**
     * A wrapper for the {@link #enter(Class, Object, Enum)} and {@link #exit(String)} primitives for a {@link Runnable}.
     * @param probe the configuration class. Must not be {@code null}.
     * @param name the name of the recorded payload. If {@code null}, the payload recording will be discarded.
     * @param controlObject the control object this payload belongs to or {@code null}. If control objects should be recorded you have to override {@link PayloadProbe#isControlObjects()}.
     * @param type the payload type. Must by an object of your custom enum class registered by overriding {@link PayloadProbe#getCustomTypes()} or {@code null}.
     * @param runnable the runnable that will be timed for the payload recording.
     */
    public static void execute(Class<? extends PayloadProbe> probe, String name, Object controlObject, Enum<?> type, Runnable runnable) {
        enter(probe, controlObject, type);
        try {
            runnable.run();
        } finally {
            exit(name);
        }
    }

    /**
     * A wrapper for the {@link #enter(Class)} and {@link #exit(String)} primitives for a {@link Callable}.
     * @param probe the configuration class. Must not be {@code null}.
     * @param name the name of the recorded payload. If {@code null}, the payload recording will be discarded.
     * @param callable the callable that will be timed for the payload recording.
     */
    public static <T> T execute(Class<? extends PayloadProbe> probe, String name, Callable<T> callable) throws Exception {
        return execute(probe, name, null, null, callable);
    }

    /**
     * A wrapper for the {@link #enter(Class, Object, Enum)} and {@link #exit(String)} primitives for a {@link Callable}.
     * @param probe the configuration class. Must not be {@code null}.
     * @param name the name of the recorded payload. If {@code null}, the payload recording will be discarded.
     * @param controlObject the control object this payload belongs to or {@code null}. If control objects should be recorded you have to override {@link PayloadProbe#isControlObjects()}.
     * @param type the payload type. Must by an object of your custom enum class registered by overriding {@link PayloadProbe#getCustomTypes()} or {@code null}.
     * @param callable the callable that will be timed for the payload recording.
     */
    public static <T> T execute(Class<? extends PayloadProbe> probe, String name, Object controlObject, Enum<?> type, Callable<T> callable) throws Exception {
        enter(probe, controlObject, type);
        try {
            return callable.call();
        } finally {
            exit(name);
        }
    }

    /**
     * Records an open event for a control object and assigns a name to it. The control object will be displayed in the control objects view and the timeline.
     * Your configuration subclass of {@link PayloadProbe} must override {@link PayloadProbe#isControlObjects()} and return {@code true} to enable control object
     * recording as a capability for the probe.
     * <p>
     * If you don't use this method and a previously unknown control object is used in the {@code enter} or {@code execute} methods and
     * you have overridden the {@link PayloadProbe#getControlObjectName(Object)} method of your configuration class, it will be called to assign a name to the new control object.
     * No open event will be created in the events view in that case.
     * </p>
     *
     * @param probe the configuration class. Must not be {@code null}.
     * @param controlObject the control object
     * @param name the name of the control object
     */
    public static void openControlObject(Class<? extends PayloadProbe> probe, Object controlObject, String name) {
    }

    /**
     * Closes a control object. If the control object is not currently open, this method has no effect.
     * @param probe the configuration class. Must not be {@code null}.
     * @param controlObject the control object
     */
    public static void closeControlObject(Class<? extends PayloadProbe> probe, Object controlObject) {
    }

    /**
     * You can register your probe independently of {@code enter} and {@code execute} calls. As soon as the probe is registered
     * it will be displayed in the JProfiler UI and telemetries can be recorded.
     * You don't have to call this method if you don't want to start telemetry recording before an {@code enter} or {@code execute} call
     * takes place. {@code enter} or {@code execute} and all control object methods will automatically register your probe.
     *
     * @param probe the configuration class. Must not be {@code null}.
     */
    public static void register(Class<? extends PayloadProbe> probe) {
    }

    static {
        JProfilerEmbeddedPackageMarker.register();
    }
}
