/*
 * Copyright 2025 ej-technologies GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 */
package com.jprofiler.api.controller;

/**
 * Request tracking options for calling {@link Controller#startCPURecording(boolean, TrackingOptions)}} and
 * {@link Controller#startAllocRecording(boolean, TrackingOptions)}. See the help topic on request
 * tracking for more information.
 * After constructing an instance of this class, all request tracking options are set to {@code false}.
 */
public class TrackingOptions {

    /**
     * The default value used by {@link Controller#startCPURecording(boolean)} and {@link Controller#startAllocRecording(boolean, TrackingOptions)}
     */
    public static final TrackingOptions NONE = new TrackingOptions().freeze();

    /**
     * Only track platform thread start, same as {@code new TrackingOptions().platformThreadStart(true)}.
     */
    public static final TrackingOptions PLATFORM_THREAD = new TrackingOptions().platformThread(true).freeze();

    /**
     * Only track executors, same as {@code new TrackingOptions().executors(true)}.
     */
    public static final TrackingOptions EXECUTORS = new TrackingOptions().executors(true).freeze();

    /**
     * Only track AWT, same as {@code new TrackingOptions().awt(true)}.
     */
    public static final TrackingOptions AWT = new TrackingOptions().awt(true).freeze();

    /**
     * Only track SWT, same as {@code new TrackingOptions().swt(true)}.
     */
    public static final TrackingOptions SWT = new TrackingOptions().swt(true).freeze();

    /**
     * Only track RMI, same as {@code new TrackingOptions().rmi(true)}.
     */
    public static final TrackingOptions RMI = new TrackingOptions().rmi(true).freeze();

    /**
     * Only track gRPC, same as {@code new TrackingOptions().grpc(true)}.
     */
    public static final TrackingOptions GRPC = new TrackingOptions().grpc(true).freeze();

    /**
     * Only track remote EJBs, same as {@code new TrackingOptions().remoteEjb(true)}.
     */
    public static final TrackingOptions REMOTE_EJB = new TrackingOptions().remoteEjb(true).freeze();

    /**
     * Only track http requests, same as {@code new TrackingOptions().http(true)}.
     */
    public static final TrackingOptions HTTP = new TrackingOptions().http(true).freeze();

    /**
     * Only track kotlin coroutines, same as {@code new TrackingOptions().kotlinCoroutines(true)}.
     */
    public static final TrackingOptions KOTLIN_COROUTINES = new TrackingOptions().kotlinCoroutines(true).freeze();

    /**
     * Only track virtual thread start, same as {@code new TrackingOptions().virtualThread(true)}.
     */
    public static final TrackingOptions VIRTUAL_THREAD = new TrackingOptions().virtualThread(true).freeze();

    private boolean platformThread = false;
    private boolean virtualThread = false;
    private boolean executors = false;
    private boolean awt = false;
    private boolean swt = false;
    private boolean rmi  = false;
    private boolean grpc  = false;
    private boolean remoteEjb = false;
    private boolean http = false;
    private boolean kotlinCoroutines = false;

    private boolean frozen = false;

    /**
     * Create a new instance with no tracking types enabled.
     */
    public TrackingOptions() {
    }

    /**
     * Determines if platform thread start should be tracked.
     * @param threadStart the new value
     * @return this instance
     */
    public TrackingOptions platformThread(boolean threadStart) {
        checkFrozen();
        this.platformThread = threadStart;
        return this;
    }

    /**
     * Determines if virtual thread start should be tracked.
     * @param threadStart the new value
     * @return this instance
     */
    public TrackingOptions virtualThread(boolean threadStart) {
        checkFrozen();
        this.virtualThread = threadStart;
        return this;
    }

    /**
     * Determines if executors should be tracked.
     * @param executors the new value
     * @return this instance
     */
    public TrackingOptions executors(boolean executors) {
        checkFrozen();
        this.executors = executors;
        return this;
    }

    /**
     * Determines if AWT should be tracked.
     * @param awt the new value
     * @return this instance
     */
    public TrackingOptions awt(boolean awt) {
        checkFrozen();
        this.awt = awt;
        return this;
    }

    /**
     * Determines if SWT should be tracked.
     * @param swt the new value
     * @return this instance
     */
    public TrackingOptions swt(boolean swt) {
        checkFrozen();
        this.swt = swt;
        return this;
    }

    /**
     * Determines if RMI should be tracked.
     * @param rmi the new value
     * @return this instance
     */
    public TrackingOptions rmi(boolean rmi) {
        checkFrozen();
        this.rmi = rmi;
        return this;
    }

    /**
     * Determines if gRPC should be tracked.
     * @param grpc the new value
     * @return this instance
     */
    public TrackingOptions grpc(boolean grpc) {
        checkFrozen();
        this.grpc = grpc;
        return this;
    }

    /**
     * Determines if http requests should be tracked.
     * @param http the new value
     * @return this instance
     */
    public TrackingOptions http(boolean http) {
        checkFrozen();
        this.http = http;
        return this;
    }

    /**
     * Determines if remote EJBs should be tracked.
     * @param remoteEjb the new value
     * @return this instance
     */
    public TrackingOptions remoteEjb(boolean remoteEjb) {
        checkFrozen();
        this.remoteEjb = remoteEjb;
        return this;
    }

    /**
     * Determines if kotlin coroutines should be tracked.
     * @param kotlinCoroutines the new value
     * @return this instance
     */
    public TrackingOptions kotlinCoroutines(boolean kotlinCoroutines) {
        checkFrozen();
        this.kotlinCoroutines = kotlinCoroutines;
        return this;
    }

    /**
     * Returns if platform thread start is tracked.
     * @return the value
     */
    public boolean isPlatformThread() {
        return platformThread;
    }

    /**
     * Returns if virtual thread start is tracked.
     * @return the value
     */
    public boolean isVirtualThread() {
        return virtualThread;
    }

    /**
     * Returns if executors are tracked.
     * @return the value
     */
    public boolean isExecutors() {
        return executors;
    }

    /**
     * Returns if AWT is tracked.
     * @return the value
     */
    public boolean isAwt() {
        return awt;
    }

    /**
     * Returns if SWT is tracked.
     * @return the value
     */
    public boolean isSwt() {
        return swt;
    }

    /**
     * Returns if RMI is tracked.
     * @return the value
     */
    public boolean isRmi() {
        return rmi;
    }

    /**
     * Returns if gRPC is tracked.
     * @return the value
     */
    public boolean isGrpc() {
        return grpc;
    }

    /**
     * Returns if http requests are tracked.
     * @return the value
     */
    public boolean isHttp() {
        return http;
    }

    /**
     * Returns if remote EJBs are tracked.
     * @return the value
     */
    public boolean isRemoteEjb() {
        return remoteEjb;
    }

    /**
     * Returns if kotlin coroutines are tracked.
     * @return the value
     */
    public boolean isKotlinCoroutines() {
        return kotlinCoroutines;
    }

    /**
     * Make these tracking options unmodifiable. Further modification attempts will result in a
     * {@code UnsupportedOperationException}.
     * @return this instance
     */
    public TrackingOptions freeze() {
        frozen = true;
        return this;
    }

    private void checkFrozen() {
        if (frozen) {
            throw new UnsupportedOperationException("parameters are frozen");
        }
    }
}
