package com.jprofiler.api.agent.heap;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;

/**
 * A descriptor for a class, interface, array, or primitive type.
 */
public interface ClassDescriptor {
    /**
     * Retrieves a unique identifier of the class.
     *
     * @return a long value representing the unique identifier
     */
    long id();

    /**
     * Retrieves the display name as shown in JProfiler of the type represented by this descriptor.
     *
     * @return a non-null string representing the display name of the type.
     */
    @NotNull
    String displayName();

    /**
     * Returns the field descriptor for a class,
     * interface, array, or primitive type according to the Java Virtual Machine Specification.
     *  A field type descriptor string for a non-array type is either a one-letter code corresponding to
     *  a primitive type ({@code "J", "I", "C", "S", "B", "D", "F", "Z", "V"}), or the letter {@code "L"}, followed
     * by the fully qualified binary name of a class, followed by {@code ";"}.
     * A field type descriptor for an array type is the character {@code "["}
     * followed by the field descriptor for the component type.  Examples of
     * valid type descriptor strings include {@code "Ljava/lang/String;"}, {@code "I"},
     * {@code "[I"}, {@code "V"}, {@code "[Ljava/lang/String;"}, etc.
     *
     * @return a non-null string representing the field descriptor of the type.
     */
    @NotNull
    String descriptor();

    /**
     * Determines if the type represented by this descriptor is an array of objects.
     *
     * @return true if the type is an array of objects, false otherwise.
     */
    boolean isObjectArray();

    /**
     * Determines if the type represented by this descriptor is an array of a primitive type.
     *
     * @return true if the type is an array of a primitive type, false otherwise.
     */
    boolean isPrimitiveArray();

    /**
     * Determines if this descriptor represents {@code java.lang.String}.
     *
     * @return true if java.lang.String, false otherwise.
     */
    boolean isString();

    /**
     * Determines if this descriptor represents a {@code java.lang.Class} object in the heap dump.
     *
     * @return true if the descriptor represents a class object, false otherwise.
     */
    boolean isClassObject();

    /**
     * Determines if this descriptor represents an instance of a class in the heap dump.
     *
     * @return true if the descriptor represents a class instance, false otherwise.
     */
    boolean isInstance();

    /**
     * Retrieves the descriptor representing the superclass of the class described
     * by this descriptor.
     *
     * @return the {@link ClassDescriptor} of the superclass, or {@code null} if no
     *         superclass exists or is applicable.
     */
    @Nullable
    ClassDescriptor superClass();

    /**
     * Retrieves a list of declared instance fields for the class described by this descriptor.
     * Declared instance fields are the fields directly defined in this class, excluding those inherited
     * from superclasses or static fields.
     * <p>
     *     If field information is unavailable, the list is empty.
     * </p>
     *
     * @return a non-null list of strings where each string represents the name of an instance field
     *         declared in this class.
     */
    @NotNull
    List<String> declaredInstanceFields();

    /**
     * Retrieves a list of all instance fields present in the class described by this descriptor.
     * This list includes both declared fields and inherited instance fields from superclasses.
     * Static fields are excluded.
     * <p>
     *     If field information is unavailable, the list is empty.
     * </p>
     *
     * @return a non-null list of strings where each string represents the name of an instance field
     *         available in this class and its superclasses.
     */
    @NotNull
    List<String> allInstanceFields();

    /**
     * Retrieves a list of static field names for the class described by this descriptor.
     * <p>
     *     If field information is unavailable, the list is empty.
     * </p>
     *
     * @return a non-null list of strings where each string represents the name of a static field
     *         declared in this class.
     */
    @NotNull
    List<String> staticFields();
}
