package com.jprofiler.api.agent;

/**
 * Represents a proxy to a Kafka {@code ConsumerRecord}.
 * Provides access to the topic, partition, offset, key, value, and other metadata of a Kafka record.
 * Acts as an abstraction layer to work with Kafka {@code ConsumerRecord} objects.
 */
public interface KafkaConsumerRecord {

    /**
     * Gets the topic to which the record belongs.
     *
     * @return the name of the Kafka topic.
     */
    String topic();

    /**
     * Gets the partition number from which the record was received.
     *
     * @return the partition ID of the Kafka topic.
     */
    int partition();

    /**
     * Gets the offset of the record in the Kafka topic partition.
     *
     * @return the offset of the record.
     */
    int offset();

    /**
     * Gets the deserialized key of the record.
     *
     * @return the key of the record, or {@code null} if no key is present.
     */
    Object key();

    /**
     * Gets the deserialized value of the record.
     *
     * @return the value of the record, or {@code null} if no value is present.
     */
    Object value();

    /**
     * Gets the value of the last header with the specified key from the record's headers.
     *
     * @param key the key of the desired header.
     * @return the value of the last header with the given key as a byte array,
     * or {@code null} if the header does not exist.
     */
    byte[] lastHeaderValue(String key);

    /**
     * Gets the size of the serialized key, in bytes.
     *
     * @return the size of the serialized key, or {@code -1} if the key is not present.
     */
    int serializedKeySize();

    /**
     * Gets the size of the serialized value, in bytes.
     *
     * @return the size of the serialized value, or {@code -1} if the value is not present.
     */
    int serializedValueSize();

    /**
     * Gets the underlying consumer record represented by this proxy.
     * This method returns the original Kafka {@code ConsumerRecord} object.
     *
     * @return the original consumer record
     */
    Object consumerRecord();
}