/*
 * Copyright 2026 ej-technologies GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 */
package com.jprofiler.api.probe.injected.interception;

import com.jprofiler.api.probe.injected.Payload;
import com.jprofiler.api.probe.injected.ProbeContext;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * Declares an interception that creates a payload. The annotated probe method must be public and static and return {@code String} or {@link Payload}.
 * <p>You can get information from the intercepted method like parameter values, the instance or the return value by declaring parameters annotated with
 * the annotations from the <a href="{@docRoot}/com/jprofiler/api/probe/injected/parameter/package-summary.html">com.jprofiler.api.probe.injected.parameter</a> package. You can also declare a parameter of type {@link ProbeContext} to
 * open or close control objects or create payloads with enhanced functionality.</p>
 * <p>If you return a {@link Payload} to specify control objects or custom event types you must declare a parameter of type {@link ProbeContext} and create the returned object with
 * {@link ProbeContext#createPayload(String, Object, Enum)} or {@link ProbeContext#createPayloadWithDeferredName(Object, Enum)}. The {@link #invokeOn} argument must be set
 * to {@link InvocationType#ENTER} in that case.</p>
 * <p>If you create a {@link Payload} with a deferred name because the payload name cannot be determined at method entry, you must declare a method directly below the current interception method,
 * annotate it with {@link AdditionalInterception} where {@link AdditionalInterception#value()} is set to {@link InvocationType#EXIT} and return a {@code String} from that method.</p>
 * <p>If you don't want to create a payload for the current invocation, the probe method can return {@code null}.</p>
 *
 */
@Retention(RetentionPolicy.RUNTIME)
@Target(ElementType.METHOD)
public @interface PayloadInterception {
    /**
     * The point at which the annotated probe method is invoked.
     * @return the invocation type
     */
    InvocationType invokeOn();

    /**
     * Specifies the method that should be intercepted.
     * @return a method specification
     */
    MethodSpec method();

    /**
     * Indicates if your interception is reentrant. If {@code false}, the probe method will not be called for recursive invocations of the intercepted method.
     * <p>If {@code true}, the payload name (and the other information in {@link Payload}) returned by the subsequent inner calls will be used, but timing and call tree location will be used from the outermost call. You can use
     * {@link ProbeContext#restartTiming()} and {@link ProbeContext#restartTimingAndLocation()} to customize this behavior. If you want to retain the name from outer calls you can use {@link ProbeContext#getOuterPayload()}.</p>
     *
     * @return if your probe method should be called for recursive invocations
     */
    boolean reentrant() default false;
    /**
     * If payloads should also be recorded if the method throws an exception.
     * If {@link #invokeOn()} is set to {@link InvocationType#EXCEPTION} or {@link InvocationType#EXIT},
     * JProfiler will always record payloads for exceptional exits.
     *
     * @return if payloads should be recorded for exceptional exits
     */
    boolean exceptionPayloads() default false;
}
