/*
 * Copyright 2026 ej-technologies GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 */
package com.jprofiler.api.probe.injected;

import com.jprofiler.api.probe.injected.interception.*;

/**
 * An interface for enhanced probe functionality. In methods annotated with {@link Interception}, {@link PayloadInterception}, {@link SplitInterception} and {@link AdditionalInterception},
 * you can declare a parameter of this type in any position in order to get an instance of this class.
 */
public interface ProbeContext {
    /**
     * Records an open event for a control object and assigns a name to it. The control object will be displayed in the control objects view and the timeline.
     * You must set {@link Probe#controlObjects()} to {@code true} to enable control object recording as a capability for the probe.
     * <p>
     * If you don't use this method and a previously unknown control object is used in a {@link #createPayload(String, Object, Enum)} call and
     * you have annotated a method in your probe class with {@link ControlObjectName}, that method will be called to create a name for the new control object.
     * No open event will be created in the events view in that case.
     * </p>
     *
     * @param controlObject the control object
     * @param name          the name of the control object
     */
    void openControlObject(Object controlObject, String name);

    /**
     * Closes a control object. If the control object is not currently open, this method has no effect.
     *
     * @param controlObject the control object
     */
    void closeControlObject(Object controlObject);

    /**
     * <p>
     * Creates an enhanced payload object with type and control object info. You can use the created object in
     * </p>
     * <ul>
     * <li>a method returning a {@link Payload} that is either annotated with {@link PayloadInterception} and
     * where {@link PayloadInterception#invokeOn()} is set to {@link InvocationType#ENTER}
     * </li>
     * <li>
     * a method with {@link SplitInterception} and {@link SplitInterception#payloads()} set to {@code true}
     * </li>
     * </ul>
     * <p>
     * If you don't want to assign states or control objects, you can just return the name as a {@code String} directly.
     * </p>
     *
     * @param name          the name of the recorded payload. If {@code null}, the payload recording will be discarded.
     * @param type          the payload type. Must by an object of your custom enum class registered with {@link Probe#customTypes()} or {@code null}.
     * @param controlObject the control object this payload belongs to or {@code null}. If control objects should be recorded, you have to set {@link Probe#controlObjects()} to {@code true}.
     * @return a {@link Payload} instance to be returned by your interception method
     */
    Payload createPayload(String name, Object controlObject, Enum<?> type);

    /**
     * Creates an enhanced payload object with type and control object info which name you don't know yet. You can use
     * the created object in a method returning a {@link Payload} that is annotated with {@link PayloadInterception} and
     * where {@link PayloadInterception#invokeOn()} is set to {@link InvocationType#ENTER}. If you don't want to assign
     * states or control objects you can just return the name as a {@code String} directly.
     * <p>
     * To supply the name at a later point, you have to define a method right below the current method, annotate it with {@link AdditionalInterception} and return a {@code String}.
     * </p>
     * <p>
     * If you don't want to assign states or control objects, you can set {@link PayloadInterception#invokeOn()} to {@link InvocationType#EXIT} and return the name as a {@code String} directly.
     *
     * @param type          the payload type. Must by an object of your custom enum class registered with {@link Probe#customTypes()} or {@code null}.
     * @param controlObject the control object this payload belongs to or {@code null}. If control objects should be recorded, you have to set {@link Probe#controlObjects()} to {@code true}.
     * @return a {@link Payload} instance to be returned by your interception method
     */
    Payload createPayloadWithDeferredName(Object controlObject, Enum<?> type);

    /**
     * Retrieves the payload information that has been created by an outer interception, either through recursion or by a different intercepted method.
     * <p>If your inner interception method is annotated with {@link PayloadInterception}, you can return this payload to keep the payload information unchanged. A {@link SplitInterception} won't
     * change an outer payload, so this does not apply for nested splits.
     * </p>
     *
     * @return the current payload or {@code null} if there is no current payload defined.
     */
    Payload getOuterPayload();

    /**
     * If a payload is currently recorded by an outer interception, restart timing.
     */
    void restartTiming();

    /**
     * If a payload is currently recorded by an outer interception, restart timing and set the call tree to the location of the latest {@link PayloadInterception} or the latest {@link SplitInterception} with {@link SplitInterception#payloads()} set to {@code true}.
     */
    void restartTimingAndLocation();

    /**
     * Returns the recording state of the probe.
     * <p>Methods annotated with {@link PayloadInterception} are only called when probe recording is turned on. Methods annotated with {@link SplitInterception}
     * are called when CPU or allocation recording is turned on. If {@link SplitInterception#payloads()} is set to {@code true}, they are also called when only probe recording is turned on.
     * Methods annotated with {@link Interception} are always called.
     * </p>
     *
     * @return {@code true} if your probe is being recorded
     */
    boolean isRecording();
}
