/*
 * Copyright 2026 ej-technologies GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 */
package com.jprofiler.api.probe.injected;

import com.jprofiler.api.probe.injected.interception.AdditionalInterception;
import com.jprofiler.api.probe.injected.interception.Interception;
import com.jprofiler.api.probe.injected.interception.PayloadInterception;
import com.jprofiler.api.probe.injected.interception.SplitInterception;
import com.jprofiler.api.probe.injected.telemetry.Telemetry;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * Marks a class that defines an injected probe and parametrizes the probe. The probe class will be scanned for public static methods that are annotated with
 * {@link Interception}, {@link PayloadInterception}, {@link SplitInterception}, {@link AdditionalInterception}, {@link ControlObjectName} and {@link Telemetry}.
 */
@Retention(RetentionPolicy.RUNTIME)
@Target(ElementType.TYPE)
public @interface Probe {
    /**
     * The name of the probe in the JProfiler UI.
     * In the JProfiler UI, this string is used for the view name in the view selector below the "JEE &amp; probes" section.
     * @return a name
     */
    String name();

    /**
     * An optional description of this probe.
     * It is shown in the probe view below the probe name.
     * @return the description
     */
    String description() default "";

    /**
     * Configures if this probe shows an events view in the JProfiler UI.
     * @return {@code true} if events should be shown and recorded.
     */
    boolean events() default true;

    /**
     * Configures if this probe shows the timeline and control objects views in the JProfiler UI.
     * <p>You must make use of {@link ProbeContext#createPayload(String, Object, Enum)} to associate control objects with payloads.</p>
     * @return {@code true} if control objects should be shown and recorded.
     */
    boolean controlObjects() default false;
    /**
     * Configures if this probe should record the "Events", "Average Duration" and - if control objects are recorded - the "Recorded Open Control Objects" telemetries.
     * @return {@code true} if default telemetries should be recorded.
     */
    boolean defaultTelemetries() default true;

    /**
     * You can specify custom types for events. If control objects are recorded, the same types are used for the associated control object states.
     * Specify an {@link Enum} class with all your states as constants. The {@code toString()} method is used for
     * the event display name. You can further customize your events by implementing {@link TypeCustomizer} in the enum class.
     * @return the class of your event enum
     */
    Class<? extends Enum> customTypes() default Enum.class;
}
