package com.jprofiler.api.agent.heap;

import org.jetbrains.annotations.NotNull;

import java.util.Collection;

/**
 * Represents an object from a heap dump. Provides information regarding the object's identity,
 * size, associated class description, and its relationship with other objects in the heap.
 * <p>
 * You can cast this object to its subinterfaces {@link Instance}, {@link StringInstance}, {@link PrimitiveArray}
 * and {@link ClassObject} to get further specific data. You can determine the object kind with the {@link ClassDescriptor}
 * provided by the {@link #classDescriptor()} method. For PHD snapshots, only {@link HeapObject} and {@link ClassObject} is available.
 * </p>
 *
 */
public interface HeapObject {

    /**
     * Retrieves a unique identifier of the object.
     *
     * @return a long value representing the unique identifier
     */
    long id();

    /**
     * Retrieves the shallow size of the object.
     *
     * @return the size as a long value.
     */
    long size();

    /**
     * Retrieves the class descriptor associated with this heap object. The class descriptor
     * contains metadata about the object's type, including its name, hierarchy, and fields.
     *
     * @return the {@link ClassDescriptor} instance representing the class of this heap object,
     *         never null.
     */
    @NotNull
    ClassDescriptor classDescriptor();

    /**
     * Retrieves all objects referenced directly by this heap object. The references
     * include field values, array values or other class specific referenced objects for a {@link ClassObject}.
     *
     * @return a collection of {@link HeapObject} instances that are directly referenced by this object.
     *         The collection is never null but may be empty if no objects are referenced.
     */
    @NotNull
    Collection<HeapObject> referencedObjects();

    /**
     * Retrieves all objects that directly reference this heap object.
     *
     * @return a collection of {@link HeapObject} instances that reference this object.
     *         The collection is never null but may be empty if no objects reference this heap object.
     */
    @NotNull
    Collection<HeapObject> referencingObjects();

}
