package com.jprofiler.api.agent.heap;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * Represents a {@code java.lang.Class} object in a heap dump. Provides
 * access to static field values of the class it represents.
 */
public interface ClassObject extends HeapObject {
    /**
     * Retrieves the {@link ClassDescriptor} that this {@link ClassObject} represents.
     * The descriptor provides detailed metadata about the class, including its name,
     * fields, and hierarchy.
     *
     * @return a {@link ClassDescriptor} instance representing the class described by this object, never null.
     */
    @NotNull
    ClassDescriptor representedClass();

    /**
     * Retrieves the value of a static field for this class object based on its index.
     * The static field can hold either a primitive value, a {@link HeapObject}, or one of its subinterfaces.
     *
     * @param fieldIndex the zero-based index of the static field in the class descriptor.
     *                   The index must be within the bounds of the static fields defined for the class.
     * @return the value of the static field at the specified index, or {@code null} if no field exists at
     *         the specified index, if the static fields are unavailable, or if the field value itself is {@code null}.
     */
    @Nullable
    Object staticFieldValue(int fieldIndex);

    /**
     * Retrieves the value of a static field for this class object based on its field name.
     * The static field can hold either a primitive value, a {@link HeapObject}, or one of its subinterfaces.
     *
     * @param fieldName the name of the static field to be retrieved. Must not be null or empty.
     * @return the value of the static field with the specified name, or {@code null} if no field exists
     *         with the specified name, if the static fields are unavailable, or if the field value itself is {@code null}.
     */
    @Nullable
    Object staticFieldValue(String fieldName);
}
