package com.jprofiler.api.agent;

import com.jprofiler.agent.util.HttpUtilImpl;
import org.jetbrains.annotations.NotNull;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * Class with helper methods for http splitting scripts
 */
public class HttpUtil {
    /**
     * Returns the specified part of the request URI
     * @param requestURI the request URI from {@code javax.servlet.http.HttpServletRequest.getRequestURI()}
     * @param fromSegment the first segment to be included, the index is one-based
     * @param toSegment the last segment to be included, the index is one-based
     * @param replaceNumericIds if {@code true} segments that only contain numbers are replaced with {@code <id>}
     * @return the specified part of the request URI
     */
    public static String getPathSegments(String requestURI, int fromSegment, int toSegment, boolean replaceNumericIds) {
        if (requestURI == null || requestURI.length() == 0) {
            return requestURI;
        } else {
            StringBuilder builder = new StringBuilder();
            HttpUtilImpl.addSegments(builder, requestURI, fromSegment, toSegment, replaceNumericIds, HttpUtilImpl.ID_PLACEHOLDER, false);
            return builder.toString();
        }
    }

    /**
     * Returns the value for the first occurrence of a parameter in a query string.
     * This method is intended for generic URL splitting scripts, for servlet splitting scripts the
     * {@code servletRequest.getParameter(...)} method should be used instead.
     * @param parameterName the name of the parameter
     * @param queryString the query string
     * @return the value or {@code null} if the parameter was not found
     */
    public static String getParameter(String parameterName, String queryString) {
        List<String> values = getParameters(queryString).get(parameterName);
        if (values != null && !values.isEmpty()) {
            return values.get(0);
        } else {
            return null;
        }
    }

    /**
     * Returns a map with the values for all parameters in a query string
     * This method is intended for generic URL splitting scripts, for servlet splitting scripts the
     * {@code servletRequest.getParameterMap(...)} method should be used instead.
     * @param queryString the query string
     * @return the map from parameter names to lists of parameter values
     */
    public static Map<String, List<String>> getParameters(String queryString) {
        Map<String, List<String>> keyValuePairs = new LinkedHashMap<String, List<String>>();
        String[] pairs = queryString.split("&");
        for (String pair : pairs) {
            int index = pair.indexOf("=");
            try {
                String key = index > 0 ? URLDecoder.decode(pair.substring(0, index), "UTF-8") : pair;
                List<String> values = getValues(key, keyValuePairs);
                String value = index > 0 && pair.length() > index + 1 ? URLDecoder.decode(pair.substring(index + 1), "UTF-8") : "";
                values.add(value);
            } catch (UnsupportedEncodingException ignored) {
            }
        }
        return keyValuePairs;
    }

    @NotNull
    private static List<String> getValues(String key, Map<String, List<String>> keyValuePairs) {
        List<String> values = keyValuePairs.get(key);
        if (values == null) {
            values = new ArrayList<String>();
            keyValuePairs.put(key, values);
        }
        return values;
    }
}
