/*
 * Copyright 2026 ej-technologies GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 */
package com.jprofiler.api.probe.embedded;

import com.jprofiler.api.probe.embedded.telemetry.Telemetry;

import java.util.concurrent.Callable;

/**
 * For each payload probe, you have to create a subtype of this class and pass it as a class literal
 * to the methods in {@link Payload}.
 * <p>You can override the methods of this class to configure your probe. The only method you have to implement is
 * {@link #getName()}.</p>
 * <p>You can add additional custom telemetries by adding public static methods to your subclass that return a number type and are annotated with {@link Telemetry}.</p>
 *
 */
@JProfilerEmbeddedClass
public abstract class PayloadProbe {
    /**
     * Implement this method to specify the display name of your probe
     * In the JProfiler UI, this string is used for the view name in the view selector below the "JEE &amp; probes" section.
     * @return the name of your probe
     */
    public abstract String getName();

    /**
     * You can override this method to add a description to your probe in the JProfiler UI.
     * It is shown in the probe view below the probe name.
     * @return your optional description
     */
    public String getDescription() {
        return null;
    }

    /**
     * You can override this method and return {@code false} if you want to remove the events view.
     * @return if the events view should be displayed
     */
    public boolean isEvents() {
        return true;
    }

    /**
     * You can override this method and return {@code false} if you want to remove the default telemetries.
     * The default telemetries are the event frequency and the average duration telemetries as well as the open control
     * object telemetries if {@link #isControlObjects()} returns {@code true}.
     * @return if the default telemetries will be recorded
     */
    public boolean isDefaultTelemetries() {
        return true;
    }

    /**
     * If your probe supports control objects, you have to override this method and return {@code true}.
     * @return if control objects can be recorded
     */
    public boolean isControlObjects() {
        return false;
    }

    /**
     * You can specify custom types for events. If control objects are recorded, the same types are used for the associated control object states. In the overridden method,
     * return an {@link Enum} class with all your states as constants. The {@code toString()} method is used for
     * the event display name. You can further customize your events by implementing {@link TypeCustomizer} in the enum class.
     * @return the class of your event enum
     */
    public Class<? extends Enum> getCustomTypes() {
        return null;
    }

    /**
     * If you use control objects and a previously unknown control object is passed to {@link Payload#enter(Class, Object, Enum)}, {@link Payload#execute(Class, String, Object, Enum, Runnable)} or
     * {@link Payload#execute(Class, String, Object, Enum, Callable)}, the JProfiler agent will call this method to associate a name with the control object. You can alternatively associate a name
     * with a control object by calling {@link Payload#openControlObject(Class, Object, String)}.
     *
     * @param controlObject the control object that has no name associated yet
     * @return a name that will be displayed in the JProfiler UI for this object
     */
    public String getControlObjectName(Object controlObject) {
        return null;
    }
}
