/*
 * Copyright 2026 ej-technologies GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 */
package com.jprofiler.api.agent.mbean;

/**
 * <p>
 * This is JProfiler's MBean interface. The MBean can be used with tools like JConsole or in your own application.
 * For direct calls from your code in the same JVM, rather use the {@code com.jprofiler.api.controller.Controller} class.
 * </p>
 * <b>Programmatic use</b>
 * <p>
 * To learn how to access the JProfiler MBean in a different profiled JVM programmatically, have a look at the
 * {@code $JPROFILER_HOME/api/samples/mbean} example project.
 * Open the gradle build file {@code build.gradle} in an editor for instructions on how to run it.
 * </p>
 * <b>JAR file and dependency</b>
 * <p>
 * All classes in this package are contained in {@code $JPROFILER_HOME/api/jprofiler-mbean.jar}.
 * The MBean API is also available as a dependency. In the snippets below, "VERSION"
 * should be replaced with the corresponding JProfiler version.
 * </p>
 * <b>Maven</b>
 * <pre>{@code
 * <dependency>
 *   <groupId>com.jprofiler</groupId>
 *   <artifactId>jprofiler-mbean</artifactId>
 *   <version>VERSION</version>
 * </dependency>
 * }</pre>
 * <b>Gradle</b>
 * <pre>{@code
 * dependencies {
 *   classpath group: 'com.jprofiler', name: 'jprofiler-mbean', version: 'VERSION'
 * }
 * }</pre>
 */
public interface RemoteControllerMBean {
    /**
     * Starts recording CPU data.
     * This method can be called repeatedly and alternatingly with
     * {@code stopCPURecording()}. With these methods you can restrict
     * CPU profiling to certain regions of your code.
     *
     * @param reset if {@code true}, any previously accumulated CPU profiling
     *              data will be discarded. If {@code false}, CPU data will be accumulated
     *              across pairs of invocations of  {@code startCPURecording()} and
     *              {@code stopCPURecording()}.
     */
    void startCPURecording(boolean reset);

    /**
     * Stops CPU recording.
     * This method can be called after {@code startCPURecording()} has been called.
     * However, you do not have to call it since CPU profiling can run until the
     * JVM exits.
     */
    void stopCPURecording();

    /**
     * Starts recording of memory allocations.
     * This method can be called repeatedly and alternatingly with
     * {@code stopAllocRecording()}. With these methods you can restrict
     * memory allocation profiling to certain regions of your code.
     * This is especially useful for profiling an application running within
     * an application server.
     *
     * @param reset if {@code true}, any previously recorded profiling
     *              data will be discarded. If {@code false}, allocations within all
     *              pairs of invocations of  {@code startAllocRecording()} and
     *              {@code stopAllocRecording()} will be recorded.
     */
    void startAllocRecording(boolean reset);

    /**
     * Stops recording of memory allocations.
     * This method can be called after {@code startAllocRecording()} has been called.
     * However, you do not have to call it since memory profiling can run until the
     * JVM exits.
     */
    void stopAllocRecording();

    /**
     * Adds a bookmark at the current time.
     * The bookmark will be displayed in all JProfiler graphs with a time axis.
     * The description will be displayed in the tooltip for the bookmark.
     *
     * @param description the name of the bookmark, may also be {@code null}
     */
    void addBookmark(String description);

    /**
     * Triggers a heap dump.
     * If you want to analyze a heap dump with the heap walker in a snapshot saved with
     * the {@code saveSnapshot} method, you should call this method from your source code
     * at an appropriate time.
     * <p>
     * <b>ATTENTION:</b> Taking a heap dump takes a long time (on the order of seconds).
     * If you call this method to often, your application might become unusable or take
     * an excessively long time to finish.
     * </p>
     *
     * @param fullGc                 if {@code true}, a full garbage collection will be performed.
     * @param onlyRecorded           if {@code true}, only objects recorded between startAllocRecording and
     *                               stopAllocRecording will be included in the dump.
     * @param primitiveData          if {@code true}, primitive data will also be recorded.
     * @param calculateRetainedSizes if {@code true}, the retained sizes of all objects will be calculated. This requires {@code fullGc=true}.
     */
    void triggerHeapDumpWithOptions(boolean fullGc, boolean onlyRecorded, boolean primitiveData, boolean calculateRetainedSizes);

    /**
     * Triggers a heap dump.
     * Calls triggerHeapDumpWithOptions(true, false, true, true).
     *
     * @see #triggerHeapDumpWithOptions(boolean, boolean, boolean, boolean)
     **/
    void triggerHeapDump();

    /**
     * Mark all objects currently on the heap as "old". In the next heap dump, you can then distinguish between objects created before and after this method was called.
     * This action has much less overhead than a full heap dump
     */
    void markHeap();

    /**
     * Triggers a dump of all objects on the heap to be displayed it in the "All objects" view.
     */
    void triggerAllObjectsDump();

    /**
     * Triggers a dump of all objects on the heap to be displayed it in the "All objects" view
     * with an optional label for identification.
     *
     * @param label A string parameter used to label the dump. Can be null.
     */
    void triggerAllObjectsDump(String label);

    /**
     * Triggers an MBean snapshot with an optional label. The recorded MBeans con be
     * optionally filtered with a regular expression.
     *
     * @param label An optional label for the snapshot. Can be null.
     * @param regexp An optional regular expression for filtering MBeans based on the canonical name. Can be null.
     */
    void triggerMBeanSnapshot(String label, String regexp);

    /**
     * Saves a snapshot of all profiling data to disk.
     * This is especially important for offline profiling.
     * You should choose the standard extension {@code .jps} for the
     * {@code file} parameter, since JProfiler's GUI frontend filters the corresponding
     * file choosers for that extension. If you want to save several snapshots during
     * one profiling run, take care to provide unique {@code file} parameters
     * since snapshot files will be overwritten otherwise.
     * <p>
     * <b>ATTENTION:</b> Saving a snapshot takes a long time (on the order of seconds).
     * If you call this method to often, your application might become unusable or take
     * an excessively long time to finish, and your hard disk might run out of space.
     * </p>
     *
     * @param file the file to which the snapshot should be saved.
     * @return the absolute path where the snapshot was saved. This can be in a temporary directory if the
     * working directory is not writable.
     */
    String saveSnapshot(String file);

    /**
     * Starts recording of thread states and monitor usage.
     * This method can be called repeatedly and alternatingly with
     * {@code stopThreadProfiling()}. For an offline session, thread
     * profiling is switched on by default.
     */
    void startThreadProfiling();

    /**
     * Stops recording of thread states and monitor usage.
     * This method can be called repeatedly and alternatingly with
     * {@code startThreadProfiling()}. However, you do not have to call it since
     * thread profiling can run until the JVM exits.
     */
    void stopThreadProfiling();

    /**
     * Starts recording of VM telemetry data.
     * This method can be called repeatedly and alternatingly with
     * {@code stopVMTelemetryRecording()}. For an offline session, VM telemetry
     * recording is switched on by default.
     */
    void startVMTelemetryRecording();

    /**
     * Stops recording of VM telemetry data.
     * This method can be called repeatedly and alternatingly with
     * {@code startVMTelemetryRecording()}. However, you do not have to call it since
     * VM telemetry recording can run until the JVM exits.
     */
    void stopVMTelemetryRecording();

    /**
     * Saves a snapshot of all profiling data to disk when the VM shuts down.
     * This is especially important for offline profiling.
     * You should choose the standard extension {@code .jps} for the
     * {@code file} parameter, since JProfiler's GUI frontend filters the corresponding
     * file choosers for that extension.
     * <p>
     * <b>ATTENTION:</b> Saving a snapshot can take quite some time (on the order of seconds).
     * When the VM is shut down during a user logout or a system shutdown, the OS may terminate
     * the VM before saving is completed.
     *
     * @param file the file to which the snapshot should be saved.
     */
    void saveSnapshotOnExit(String file);


    /**
     * Enable or disable all triggers with a specified group ID. The group ID can be entered in
     * the "Group ID" step of the trigger configuration wizard in the JProfiler GUI.
     *
     * @param enabled if the triggers should be enabled
     * @param groupId the group ID
     * @throws IllegalArgumentException if no trigger with the specified group ID exists
     */
    void enableTriggerGroup(boolean enabled, String groupId) throws IllegalArgumentException;

    /**
     * Enable or disable all triggers. The enabled/disabled state of the single triggers will not be lost,
     * disabling all triggers with this method overrides the enabled/disabled state of the single triggers.
     *
     * @param enabled if the triggers should be enabled
     */
    void enableTriggers(boolean enabled);

    /**
     * Starts algorithmic complexity recording for all previously configured methods.
     */
    void startComplexityRecording();

    /**
     * Stops algorithmic complexity recording and prepares data for snapshot.
     */
    void stopComplexityRecording();

    /**
     * Start recording of monitor usage with default thresholds of 100 microseconds
     * for blocking events and 100 ms for waiting events.
     * This method can be called repeatedly and alternatingly with
     * {@code stopMonitorRecording()}. Monitor profiling is switched off
     * by default.
     */
    void startMonitorRecording();

    /**
     * Start recording of monitor usage.
     * This method can be called repeatedly and alternatingly with
     * {@code stopMonitorRecording()}. Monitor profiling is switched off
     * by default.
     *
     * @param blockedThreshold the recording threshold for blocking events in microseconds.
     * @param waitingThreshold the recording threshold for waiting events in microseconds.
     */
    void startMonitorRecording(int blockedThreshold, int waitingThreshold);

    /**
     * Stop recording of monitor usage.
     * This method can be called repeatedly and alternatingly with
     * {@code startMonitorRecording()}. However, you do not have to call it since
     * monitor profiling can run until the JVM exits.
     */
    void stopMonitorRecording();

    /**
     * Trigger a thread dump.
     * @param includeVirtualThreads if {@code true}, virtual threads will be included. The thread dump won't be synchronized in this case.
     */
    void triggerThreadDump(boolean includeVirtualThreads);

    /**
     * Trigger a thread dump with an optional label for identification.
     *
     * @param includeVirtualThreads if {@code true}, virtual threads will be included. The thread dump won't be synchronized in this case.
     * @param label A string parameter used to label the dump. Can be null.
     */
    void triggerThreadDump(boolean includeVirtualThreads, String label);

    /**
     * Trigger a dump of current monitors and locks.
     */
    void triggerMonitorDump();

    /**
     * Start recording data for a selected probe.
     *
     * @param probeName        the name of the probe. For built-in probes, see the {@code PROBE_NAME} constants in {@code com.jprofiler.api.controller.Controller}.
     *                         For embedded and injected probes, the class name of the probe is used.
     *                         For script probes, this name is {@code script.n} where n is the one-based index of the script probe.
     * @param events           determines whether events should be recorded for the "Events" view of the probe. Has no effect for probes that do not have an "Events" view.
     * @param specialRecording determines whether special recordings for the probe should be switched on. For the "JDBC" probe, this controls connection leak tracking.
     *                         Has no effect for other probes without special recording.
     */
    void startProbeRecording(String probeName, boolean events, boolean specialRecording);

    /**
     * Stop recording data for a selected probe.
     *
     * @param probeName the name of the probe. For built-in probes, see the {@code PROBE_NAME} constants in {@code com.jprofiler.api.controller.Controller}.
     *                  The string values can be obtained when clicking on the "Constant Field Values" link in the Javadoc of the fields.
     *                  For embedded and injected probes, the class name of the probe is used.
     *                  For script probes, this name is {@code script.n} where n is the one-based index of the script probe.
     */
    void stopProbeRecording(String probeName);

    /**
     * Return an absolute path for the given path.
     * Relative paths will be resolved against the working directory of the profiled JVM.
     * With this method you can determine where a snapshot will be saved to when a relative path is passed to the
     * {@link #saveSnapshot(String)} method.
     * The returned path will have file separators of the remote machine.
     */
    String convertToAbsolutePath(String path);

}
