/*
 * Copyright 2026 ej-technologies GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 */
package com.jprofiler.api.controller;

/**
 * Special probe recording options for calling {@link Controller#startProbeRecording(String, ProbeRecordingOptions)}.
 * The default values after constructing an instance of this class are all set to {@code false}.
 */
public class ProbeRecordingOptions {

    /**
     * No special recordings are active. This is the same as {@link #ProbeRecordingOptions() constructing} a new object without
     * calling any further methods.
     */
    public static final ProbeRecordingOptions NONE = new ProbeRecordingOptions().freeze();

    /**
     * Same as {@link #NONE}, only with {@code events(true)}.
     */
    public static final ProbeRecordingOptions EVENTS = new ProbeRecordingOptions().events(true).freeze();

    private boolean events = false;
    private boolean connectionLeaks = false;

    private boolean frozen = false;

    /**
     * Create a new instance with the default options described above.
     */
    public ProbeRecordingOptions() {
    }

    /**
     * Determines whether single events should be recorded for the "Events" view of the probe.
     * This setting has no effect when used with a probe that does not have an event view.
     * The default is {@code false}.
     *
     * @param events the new value
     * @return this instance
     */
    public ProbeRecordingOptions events(boolean events) {
        checkFrozen();
        this.events = events;
        return this;
    }

    /**
     * Determines whether connection leaks should be recorded for the JDBC probe.
     * This setting has no effect when used with other probes.
     * The default is {@code false}.
     *
     * @param connectionLeaks the new value
     * @return this instance
     */
    public ProbeRecordingOptions connectionLeaks(boolean connectionLeaks) {
        checkFrozen();
        this.connectionLeaks = connectionLeaks;
        return this;
    }

    /**
     * Returns if single events are recorded for the "Events" view of the probe.
     *
     * @return the value
     */
    public boolean isEvents() {
        return events;
    }

    /**
     * Returns if connection leaks are recorded for the JDBC probe.
     *
     * @return the value
     */
    public boolean isConnectionLeaks() {
        return connectionLeaks;
    }

    /**
     * Make these recording options unmodifiable. Further modification attempts will result in a
     * {@code UnsupportedOperationException}.
     *
     * @return this instance
     */
    public ProbeRecordingOptions freeze() {
        frozen = true;
        return this;
    }

    private void checkFrozen() {
        if (frozen) {
            throw new UnsupportedOperationException("parameters are frozen");
        }
    }
}
