/*
 * Copyright 2026 ej-technologies GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 */
package com.jprofiler.api.controller;

/**
 * Heap dump options for calling {@link Controller#triggerHeapDump(HeapDumpOptions)}
 * The default values after constructing an instance of this class are equivalent to calling
 * {@code heapDumpOptions.fullGc(true).primitiveData(true).calculateRetainedSizes(true).selectRecorded(true)}.
 */
public class HeapDumpOptions {
    private static final boolean DEFAULT_PRIMITIVE_DATA = !isJ9VM(); // is used by other static fields and must be initialized first
    
    /**
     * The default value used by {@link Controller#triggerHeapDump()}. This uses the default options described above.
     */
    public static final HeapDumpOptions DEFAULT = new HeapDumpOptions().freeze();

    /**
     * Same as {@link #DEFAULT}, only with {@code selectRecorded(true)}.
     */
    public static final HeapDumpOptions SELECT_RECORDED = new HeapDumpOptions().selectRecorded(true).freeze();

    /**
     * Same as {@link #DEFAULT}, only with {@code fullGc(false)}.
     */
    public static final HeapDumpOptions NO_FULL_GC = new HeapDumpOptions().fullGc(false).freeze();

    private boolean fullGc = true;
    private boolean primitiveData = DEFAULT_PRIMITIVE_DATA;
    private boolean calculateRetainedSizes = true;
    private boolean selectRecorded = false;

    private boolean retainSoftReferences = true;
    private boolean retainWeakReferences = false;
    private boolean retainPhantomReferences = false;
    private boolean retainFinalizerReferences = false;

    private boolean frozen = false;

    /**
     * Create a new instance with the default options described above.
     */
    public HeapDumpOptions() {
    }

    /**
     * Determines whether a full garbage collection should be performed for the heap dump.
     * The default is {@code true}.
     * @param fullGc the new value
     * @return this instance
     */
    public HeapDumpOptions fullGc(boolean fullGc) {
        checkFrozen();
        this.fullGc = fullGc;
        return this;
    }

    /**
     * Determines whether primitive data should be recorded for the heap dump.
     * The default is {@code true}.
     * @param primitiveData the new value
     * @return this instance
     */
    public HeapDumpOptions primitiveData(boolean primitiveData) {
        checkFrozen();
        this.primitiveData = primitiveData;
        return this;
    }

    /**
     * Determines whether retained sizes should be calculated for the heap dump.
     * The default is {@code true}.
     * @param calculateRetainedSizes the new value
     * @return this instance
     */
    public HeapDumpOptions calculateRetainedSizes(boolean calculateRetainedSizes) {
        checkFrozen();
        this.calculateRetainedSizes = calculateRetainedSizes;
        return this;
    }

    /**
     * Determines whether recorded objects should be selected when the heap dump is opened in the JProfiler GUI.
     * The default value is {@code false}.
     * @param selectRecorded the new value
     * @return this instance
     */
    public HeapDumpOptions selectRecorded(boolean selectRecorded) {
        checkFrozen();
        this.selectRecorded = selectRecorded;
        return this;
    }

    /**
     * Determines whether soft references should be considered for full GC. The default value is {@code true}.
     * To remove objects that are only held by soft references, call this method with the argument {@code false}.
     * @param retainSoftReferences whether soft references should be retained in the heap dump
     * @return this instance
     * @see #fullGc
     */
    public HeapDumpOptions retainSoftReferences(boolean retainSoftReferences) {
        checkFrozen();
        this.retainSoftReferences = retainSoftReferences;
        return this;
    }

    /**
     * Determines whether weak references should be considered for full GC. The default value is {@code false}.
     * To retain objects that are only held by weak references, call this method with the argument {@code true}.
     * @param retainWeakReferences whether weak references should be retained in the heap dump
     * @return this instance
     * @see #fullGc
     */
    public HeapDumpOptions retainWeakReferences(boolean retainWeakReferences) {
        checkFrozen();
        this.retainWeakReferences = retainWeakReferences;
        return this;
    }

    /**
     * Determines whether phantom references should be considered for full GC. The default value is {@code false}.
     * To retain objects that are only held by phantom references, call this method with the argument {@code true}.
     * @param retainPhantomReferences whether phantom references should be retained in the heap dump
     * @return this instance
     * @see #fullGc
     */
    public HeapDumpOptions retainPhantomReferences(boolean retainPhantomReferences) {
        checkFrozen();
        this.retainPhantomReferences = retainPhantomReferences;
        return this;
    }

    /**
     * Determines whether finalizer references should be considered for full GC. The default value is {@code false}.
     * To retain objects that are only held by finalizer references, call this method with the argument {@code true}.
     * @param retainFinalizerReferences whether finalizer references should be retained in the heap dump
     * @return this instance
     * @see #fullGc
     */
    public HeapDumpOptions retainFinalizerReferences(boolean retainFinalizerReferences) {
        checkFrozen();
        this.retainFinalizerReferences = retainFinalizerReferences;
        return this;
    }

    /**
     * Returns if a full garbage collection should be performed for the heap dump.
     * @return the value
     */
    public boolean isFullGc() {
        return fullGc;
    }

    /**
     * Returns if primitive data should be recorded for the heap dump.
     * @return the value
     */
    public boolean isPrimitiveData() {
        return primitiveData;
    }

    /**
     * Returns if retained sizes should be calculated for the heap dump.
     * @return the value
     */
    public boolean isCalculateRetainedSizes() {
        return calculateRetainedSizes;
    }

    /**
     * Returns if recorded objects should be selected when the heap dump is opened in the JProfiler GUI.
     * @return the value
     */
    public boolean isSelectRecorded() {
        return selectRecorded;
    }

    /**
     * Returns if soft references should retain objects in heap dump if full GC is selected.
     * @return the value
     * @see #fullGc
     */
    public boolean isRetainSoftReferences() {
        return retainSoftReferences;
    }

    /**
     * Returns if weak references should retain objects in heap dump if full GC is selected.
     * @return the value
     * @see #fullGc
     */
    public boolean isRetainWeakReferences() {
        return retainWeakReferences;
    }

    /**
     * Returns if phantom references should retain objects in heap dump if full GC is selected.
     * @return the value
     * @see #fullGc
     */
    public boolean isRetainPhantomReferences() {
        return retainPhantomReferences;
    }

    /**
     * Returns if finalizer references should retain objects in heap dump if full GC is selected.
     * @return the value
     * @see #fullGc
     */
    public boolean isRetainFinalizerReferences() {
        return retainFinalizerReferences;
    }

    /**
     * Make these heap dump options unmodifiable. Further modification attempts will result in a
     * {@code UnsupportedOperationException}.
     * @return this instance
     */
    public HeapDumpOptions freeze() {
        frozen = true;
        return this;
    }

    private void checkFrozen() {
        if (frozen) {
            throw new UnsupportedOperationException("parameters are frozen");
        }
    }

    private static boolean isJ9VM() {
        String vmVendor = System.getProperty("java.vm.vendor", "").toLowerCase();
        return vmVendor.contains("ibm") || vmVendor.contains("openj9");
    }
}
