/*
 * Copyright 2026 ej-technologies GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 */
package com.jprofiler.api.probe.embedded;

/**
 * Allows you to override the thread state recorded in JProfiler.
 * <p>You can nest calls to {@link #enter()} and {@link #exit()}, but only the outermost call is used for recording.</p>
 * <p>It is very important that a matching {@link #exit()} is always called, so the exit call must be put it a finally block like this:</p>
 * <pre>{@code
 *     ThreadState.NETIO.enter();
 *     try {
 *         // your code
 *     } finally {
 *         ThreadState.exit();
 *     }
 * }</pre>
 */
@JProfilerEmbeddedClass
public enum ThreadState {
    /**
     * The current thread is runnable.
     */
    RUNNABLE,
    /**
     * The current thread is waiting.
     */
    WAITING,
    /**
     * The current thread is blocked on a monitor.
     */
    BLOCKED,
    /**
     * The current thread is in a Net IO operation.
     */
    NETIO;

    /**
     * Enters a thread state. This state will be recorded until the matching exit is called.
     */
    public void enter() {
    }

    /**
     * Exists the current override state and resets the state to the value reported by the JVM.
     */
    public static void exit() {
    }

    static {
        JProfilerEmbeddedPackageMarker.register();
    }
}
