/*
 * Copyright 2025 ej-technologies GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 */
package com.jprofiler.api.probe.embedded.telemetry;

import com.jprofiler.api.probe.embedded.PayloadProbe;
import com.jprofiler.api.probe.embedded.SplitProbe;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * Create a custom telemetry from the numeric return value of the annotated static method in your subclasses of {@link PayloadProbe} or {@link SplitProbe}.
 * <p>
 * Only public static parameterless methods can be used for creating custom telemetries, annotating instance methods or non-public
 * static methods will not have any effect.
 * The method must return a numeric value, either a primitive value like {@code int} or {@code double}, a
 * {@code java.math.BigDecimal} or a primitive wrapper instance like {@code java.lang.Integer} or {@code java.lang.Double}.
 * </p>
 * <p>
 * The method is called periodically on a dedicated thread. This may have implications for the requirements of
 * <b>thread safety</b> in your code. If the numeric value is calculated on the fly, access to the data structures
 * must by made thread-safe by making access to those data structures synchronized. Alternatively, you could
 * calculate a cached value when convenient and save it to a volatile atomic value, like an {@code int} or a
 * class from the {@code java.util.concurrent.atomic} package.
 * </p>
 * <p>
 * For example, the following code snippet defines a custom telemetry with the name "Connection count":
 * </p>
 * <pre>
 *     &#064;Telemetry("Connection count")
 *     public static int getConnectionCount() {
 *         return connectionCount;
 *     }</pre>
 * <p>
 * <b>Multiple lines in a single custom telemetry</b>
 * </p>
 * <p>
 * If you want to compare several measurements in the same telemetry, you can annotate multiple methods with the
 * same {@link #value()} for the telemetry name, but with a different {@link #line()} parameter.
 * </p>
 * <p>
 * For example:
 * </p>
 * <pre>
 *     &#064;Telemetry(value = "Queries", line = "Successful queries")
 *     public static int getSuccessCount() {
 *         return successCount;
 *     }
 *     &#064;Telemetry(value = "Queries", line = "Failed queries")
 *     public static int getErrorCount() {
 *         return errorCount;
 *     }</pre>
 * <p>
 * This will create a single custom telemetry with the name "Queries" and two data lines named
 * "Successful queries" and "Failed queries".
 * </p>
 */
@Retention(RetentionPolicy.RUNTIME)
@Target(ElementType.METHOD)
public @interface Telemetry {
    /**
     * The name of the telemetry. Give different names to different telemetries. Reuse the same name when annotating
     * different methods if you are creating a custom telemetry with multiple lines.
     * If only a single line is used, you can omit the "value" parameter name, as in
     * <pre>
     *     &#064;Telemetry("Connection count")</pre>
     */
    String value();

    /**
     * The optional line name of the telemetry. Only specify this parameter if you're creating a telemetry with
     * multiple lines. See {@link Telemetry} for an example.
     */
    String line() default "";

    /**
     * Optional display options for the telemetry.
     */
    TelemetryFormat format() default @TelemetryFormat;
}
