/*
 * Copyright 2025 ej-technologies GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 */
package com.jprofiler.api.probe.embedded;

import com.jprofiler.api.probe.embedded.telemetry.Telemetry;

/**
 * For each split probe, you have to create a subtype of this class and pass it as a class literal
 * to the methods in {@link Split}. You can override the methods of this class to configure your probe.
 * <p>If you return {@code true} in {@link #isPayloads()} you can add additional custom telemetries by adding public
 * static methods to your subclass that return a number type and are annotated with {@link Telemetry}.</p>
 */
@JProfilerEmbeddedClass
public abstract class SplitProbe {
    /**
     * If you want to create nested splits with this probe you can override this method and return {@code true}. Otherwise, only the
     * top level call in the stack to the methods in {@link Split} will have an effect.
     * @return if nested splits will be recorded
     */
    public boolean isReentrant() {
        return false;
    }

    /**
     * You can override this method and return {@code true} to create an entry in the probes section and record your top level
     * splits as payloads.
     * @return if the top level splits should be also recorded as payloads in the probes section
     */
    public boolean isPayloads() {
        return false;
    }

    /**
     * If you return {@code true} in {@link #isPayloads()} you can override this method to specify the display name of your probe.
     * In the JProfiler UI, this string is used for the view name in the view selector below the "JEE &amp; probes" section.
     * If you not override this method, the simple class name of the probe is used.
     * @return the name of your probe
     */
    public String getName() {
        return getClass().getSimpleName();
    }

    /**
     * If you return {@code true} in {@link #isPayloads()} you can override this method to add a description for your probe in JProfiler's probe section.
     * @return the description of your probe
     */
    public String getDescription() {
        return null;
    }

    /**
     * If you return {@code true} in {@link #isPayloads()}, but want to hide the events view you can override this method and return {@code false}.
     *
     * @return if events are recorded for payloads
     */
    public boolean isEvents() {
        return isPayloads();
    }

    /**
     * If you return {@code true} in {@link #isPayloads()} but want to remove the default payload telemetries you can override this method and return {@code false}.
     * The default telemetries are the event frequency and the average duration telemetries.
     * @return if the events view should be displayed
     */
    public boolean isDefaultTelemetries() {
        return isPayloads();
    }
}
