package com.jprofiler.api.agent;

/**
 * Context object that is passed to all scripts and provides ways to store key-value pairs for all subsequent
 * invocations of any custom script.
 * <p>
 * For example, if you need to calculate the change of an observed numeric value between script invocations, you can use
 * the context as follows:
 * <pre>{@code
 *     long previousValue = scriptContext.getLong("myValue");
 *     long currentValue = ...;
 *     long delta = currentValue - previousValue;
 *     // Use delta
 *     scriptContext.putLong("myValue", currentValue); // for the next invocation of the script
 * }</pre>
 */
public interface ScriptContext {
    /**
     * Store a key-value pair in the script context.
     * @param key the key
     * @return the value, or {@code null} if there was no mapping for the key.
     */
    Object getObject(Object key);

    /**
     * Store a key-value pair in the script context. An existing mapping is overwritten.
     * @param key the key
     * @param value the value
     * @return the value that was previously associated with specified key, or {@code null} if there was no mapping for the key.
     */
    Object putObject(Object key, Object value);

    /**
     * Store a key-value pair of value type long in the script context.
     * @param key the key
     * @return the long value, or 0 if there was no mapping
     */
    long getLong(Object key);

    /**
     * Store a key-value pair of value type long in the script context. An existing mapping is overwritten.
     * @param key the key
     * @param value the long value
     * @return the value that was previously associated with specified key, or 0 if there was no mapping for the key.
     */
    long putLong(Object key, long value);
}
