/*
 * Copyright 2025 ej-technologies GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 */
package com.jprofiler.api.probe.injected.interception;

import com.jprofiler.api.probe.injected.Payload;
import com.jprofiler.api.probe.injected.ProbeContext;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * Declares an interception that creates a split in the call tree and optionally an associated payload. The annotated probe method must be public and static and return {@code String} or {@link Payload}.
 * <p>You can get information from the intercepted method like parameter values or the instance by declaring parameters annotated with
 * the annotations from the <a href="{@docRoot}/com/jprofiler/api/probe/injected/parameter/package-summary.html">com.jprofiler.api.probe.injected.parameter</a> package. You can also declare a parameter of type {@link ProbeContext} to
 * open or close control objects or check if the probe is currently being recorded.</p>
 * <p>If you return a {@link Payload} to specify control objects or custom event types, you must declare a parameter of type {@link ProbeContext} and create the returned object with
 * {@link ProbeContext#createPayload(String, Object, Enum)}.</p>
 * <p>If you don't want to create a split for the current invocation, the probe method can return {@code null}.</p>
 *
 */
@Retention(RetentionPolicy.RUNTIME)
@Target(ElementType.METHOD)
public @interface SplitInterception {
    /**
     * Specifies the method that should be intercepted.
     * @return a method specification
     */
    MethodSpec method();

    /**
     * If payloads should be created in addition to the call tree split.
     * @return if payloads should be created.
     */
    boolean payloads() default true;

    /**
     * Indicates if your interception is reentrant. If {@code false}, the probe method will not be called for recursive invocations of the intercepted method and only one split will be created.
     * <p>If {@code true}, splits will be nested, but only one payload will be created for the outermost call.</p>
     *
     * @return if your probe method should be called for recursive invocations
     */
    boolean reentrant() default false;

    /**
     * If payloads should also be recorded if the method throws an exception.
     *
     * @return if payloads should be recorded for exceptional exits
     */
    boolean exceptionPayloads() default false;
}
