/*
 * Copyright 2025 ej-technologies GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 */
package com.jprofiler.api.probe.injected.interception;

import com.jprofiler.api.probe.injected.ProbeContext;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * Declares an interception that does not split the call tree or create a payload. The annotated probe method must be public and static and return void.
 * <p>You can get information from the intercepted method like parameter values, the instance or the return value by declaring parameters annotated with
 * the annotations from the <a href="{@docRoot}/com/jprofiler/api/probe/injected/parameter/package-summary.html">com.jprofiler.api.probe.injected.parameter</a> package. You can also declare a parameter of type {@link ProbeContext} to
 * open or close control objects or check if the probe is currently being recorded.</p>
 */
@Retention(RetentionPolicy.RUNTIME)
@Target(ElementType.METHOD)
public @interface Interception {
    /**
     * The point at which the annotated probe method is invoked.
     * @return the invocation type
     */
    InvocationType invokeOn();

    /**
     * Specifies the method that should be intercepted.
     * @return a method specification
     */
    MethodSpec method();

    /**
     * Indicates if your interception is reentrant. If {@code false}, the probe method will not be called for recursive invocations of the intercepted method.
     * @return if your probe method should be called for recursive invocations
     */
    boolean reentrant() default true;
}
