package com.install4j.api.windows;

import com.install4j.runtime.installer.platform.win32.Win32Firewall;
import org.jetbrains.annotations.NotNull;

import java.util.Collection;
import java.util.EnumSet;
import java.util.Set;

/**
 * Collection of static methods to add and delete Windows firewall rules.
 * @author ej-technologies GmbH
 */
public class WinFirewall {
    private WinFirewall() {
    }

    /**
     * Add a new firewall rule.
     * <p>This method can only be called with full administrative rights.</p>
     * @param rule the rule
     * @return {@code true} if the rule was created, {@code false} if the rule already exists.
     * @throws AccessDeniedException thrown when the action could not be performed due to missing rights.
     * @throws FirewallException thrown when another type of error was encountered.
     */
    public static boolean addRule(Rule rule) throws AccessDeniedException, FirewallException {
        int profiles = Win32Firewall.getProfiles(rule.profileTypes, rule.onlyActiveProfiles, rule.avoidPublicProfile);
        return Win32Firewall.addRule(rule.executablePath, rule.ruleName, rule.groupName, rule.ruleDirection,
                rule.networkProtocol, rule.description, rule.localPorts, rule.localAddresses, rule.remotePorts, rule.remoteAddresses, profiles, rule.enabled);
    }

    /**
     * Delete all firewall rules for a specific executable path.
     * <p>This method can only be called with full administrative rights.</p>
     * @param executablePath the executable path
     * @return the number of deleted rules.
     * @throws AccessDeniedException thrown when the action could not be performed due to missing rights.
     * @throws FirewallException thrown when another type of error was encountered.
     */
    public static int deleteRulesByExecutable(String executablePath) throws AccessDeniedException, FirewallException {
        return Win32Firewall.deleteRules(executablePath, null, null, null, null, null, null, null, null, null);
    }

    /**
     * Delete all firewall rules with a specific name.
     * <p>This method can only be called with full administrative rights.</p>
     * @param ruleName the rule name
     * @return the number of deleted rules.
     * @throws AccessDeniedException thrown when the action could not be performed due to missing rights.
     * @throws FirewallException thrown when another type of error was encountered.
     */
    public static int deleteRulesByName(String ruleName) throws AccessDeniedException, FirewallException {
        return Win32Firewall.deleteRules(null, ruleName, null, null, null, null, null, null, null, null);
    }

    /**
     * Delete all firewall rules with a specific group name.
     * <p>This method can only be called with full administrative rights.</p>
     * @param groupName the group name
     * @return the number of deleted rules.
     * @throws AccessDeniedException thrown when the action could not be performed due to missing rights.
     * @throws FirewallException thrown when another type of error was encountered.
     */
    public static int deleteRulesByGroup(String groupName) throws AccessDeniedException, FirewallException {
        return Win32Firewall.deleteRules(null, null, groupName, null, null, null, null, null, null, null);
    }

    /**
     * Delete firewall rules based on the given rule as a template. Properties that are not explicitly set for this template
     * match any value.
     * <p>This method can only be called with full administrative rights.</p>
     * @param template the template rule
     * @return the number of deleted rules.
     * @throws AccessDeniedException thrown when the action could not be performed due to missing rights.
     * @throws FirewallException thrown when another type of error was encountered.
     */
    public static int deleteRules(Rule template) throws AccessDeniedException, FirewallException {
        return Win32Firewall.deleteRules(template.executablePath, template.ruleName, template.groupName,
                template.ruleDirection, template.networkProtocol, template.description, template.localPorts, template.localAddresses, template.remotePorts, template.remoteAddresses);
    }

    /**
     * Retrieve the currently active firewall profiles.
     * @return a set of profiles
     * @throws AccessDeniedException thrown when the action could not be performed due to missing rights.
     * @throws FirewallException thrown when another type of error was encountered.
     */
    public static Set<ProfileType> getCurrentProfileTypes() throws AccessDeniedException, FirewallException {
        return Win32Firewall.translate(Win32Firewall.getCurrentProfileTypes());
    }

    /**
     * A Windows firewall rule.
     */
    public static class Rule {
        @NotNull
        private final String executablePath;
        @NotNull
        private final String ruleName;
        @NotNull
        private final RuleDirection ruleDirection;
        @NotNull
        private final NetworkProtocol networkProtocol;

        private String groupName;
        private String description;
        private String localPorts;
        private String localAddresses;
        private String remotePorts;
        private String remoteAddresses;

        private EnumSet<ProfileType> profileTypes = EnumSet.of(ProfileType.DOMAIN, ProfileType.PRIVATE);
        private boolean onlyActiveProfiles;
        private boolean avoidPublicProfile;

        private boolean enabled = true;

        /**
         * Create a rule with the minimally required parameters.
         * @param executablePath The path to the executable this rule is created for.
         * @param ruleName The name of the firewall rule.
         * @param ruleDirection If the rule is an incoming or outgoing rule.
         * @param networkProtocol If this is a rule for TCP or UDP.
         */
        public Rule(@NotNull String executablePath, @NotNull String ruleName, @NotNull RuleDirection ruleDirection, @NotNull NetworkProtocol networkProtocol) {
            this.executablePath = executablePath;
            this.ruleName = ruleName;
            this.ruleDirection = ruleDirection;
            this.networkProtocol = networkProtocol;
        }

        /**
         * An optional group name for this rule.
         * @param groupName the group name
         * @return this rule
         */
        public Rule setGroupName(String groupName) {
            this.groupName = groupName;
            return this;
        }

        /**
         * An optional description for this rule.
         * @param description the description
         * @return this rule
         */
        public Rule setDescription(String description) {
            this.description = description;
            return this;
        }

        /**
         * A list of local port numbers or port ranges separated by commas.
         * <p>Example: 80, 8080, 5000-5100</p>
         * <p>The default is any ports.</p>
         * @param localPorts the port list
         * @return this rule
         */
        public Rule setLocalPorts(String localPorts) {
            this.localPorts = localPorts;
            return this;
        }

        /**
         * One or more comma-delimited tokens specifying the local addresses from which the application can listen for traffic. Valid tokens include:
         * <ul>
         * <li>"*" or an empty value indicates any local address. If present, this must be the only token included.</li>
         * <li>A valid IPv4 or IPv6 address.</li>
         * <li>A subnet can be specified using either the subnet mask or network prefix notation. If neither a subnet mask nor a network prefix is specified, the subnet mask defaults to 255.255.255.255.</li>
         * <li>An IPv4 or IPv6 address range in the format of "start address - end address" with no spaces included.</li>
         * </ul>
         * <p>The default value is "*".</p>
         * @param localAddresses the address string
         * @return this rule
         */
        public Rule setLocalAddresses(String localAddresses) {
            this.localAddresses = localAddresses;
            return this;
        }

        /**
         * A list of remote port numbers or port ranges separated by commas.
         * <p>Example: 80, 8080, 5000-5100</p>
         * <p>The default is any ports.</p>
         * @param remotePorts the port list
         * @return this rule
         */
        public Rule setRemotePorts(String remotePorts) {
            this.remotePorts = remotePorts;
            return this;
        }

        /**
         * One or more comma-delimited tokens specifying the remote addresses from which the application can listen for traffic. Valid tokens include:
         * <ul>
         * <li>"*" or an empty value indicates any remote address. If present, this must be the only token included.</li>
         * <li>A valid IPv4 or IPv6 address.</li>
         * <li>A subnet can be specified using either the subnet mask or network prefix notation. If neither a subnet mask nor a network prefix is specified, the subnet mask defaults to 255.255.255.255.</li>
         * <li>An IPv4 or IPv6 address range in the format of "start address - end address" with no spaces included.</li>
         * <li>"Defaultgateway"</li>
         * <li>"DHCP"</li>
         * <li>"DNS"</li>
         * <li>"WINS"</li>
         * <li>"LocalSubnet" indicates any local address on the local subnet. This token is not case-sensitive.</li>
         * </ul>
         * <p>The default value is "*".</p>
         * @param remoteAddresses the address string
         * @return this rule
         */
        public Rule setRemoteAddresses(String remoteAddresses) {
            this.remoteAddresses = remoteAddresses;
            return this;
        }

        /**
         * The profiles this rule should be added for.
         * <p>The default value is {@code ProfileType.DOMAIN} and {@code ProfileType.PRIVATE}</p>
         * @param profileTypes a collection of profile types
         * @return this rule
         */
        public Rule setProfileTypes(Collection<ProfileType> profileTypes) {
            this.profileTypes = (profileTypes == null || profileTypes.isEmpty()) ? EnumSet.noneOf(ProfileType.class) : EnumSet.copyOf(profileTypes);
            return this;
        }

        /**
         * Only add for profiles that are currently active.
         * <p>The default value is {@code false}.</p>
         * @param onlyActiveProfiles the value
         * @return this object
         */
        public Rule setOnlyActiveProfiles(boolean onlyActiveProfiles) {
            this.onlyActiveProfiles = onlyActiveProfiles;
            return this;
        }

        /**
         * Only add for public profile if public profile is the only currently active profile.
         * <p>The default value is {@code false}.</p>
         * @param avoidPublicProfile the value
         * @return this object
         */
        public Rule setAvoidPublicProfile(boolean avoidPublicProfile) {
            this.avoidPublicProfile = avoidPublicProfile;
            return this;
        }

        /**
         * If this rule should be enabled when created.
         * <p>The default value is {@code true}.</p>
         * @param enabled the value
         * @return this object
         */
        public Rule setEnabled(boolean enabled) {
            this.enabled = enabled;
            return this;
        }
    }

    /**
     * The network protocol for a Windows firewall rule.
     */
    public enum NetworkProtocol {
        TCP,
        UDP
    }

    /**
     * The direction of a Windows firewall rule.
     */
    public enum RuleDirection {
        INCOMING,
        OUTGOING
    }

    /**
     * The Windows firewall profiles.
     */
    public enum ProfileType {
        DOMAIN,
        PRIVATE,
        PUBLIC
    }

    /**
     * Thrown if a modification cannot be performed because the calling process does not have administrative rights.
     */
    public static class AccessDeniedException extends Exception {
    }

    /**
     * Thrown if an error other than missing access rights is encountered.
     */
    public static class FirewallException extends Exception {
        /**
         * Creates a new exception.
         * @param message the message
         */
        public FirewallException(String message) {
            super(message);
        }
    }
}
