package com.install4j.api.update;

import java.util.Objects;

/**
 * Enumeration class that represents the different update schedules. These values are used in the
 * {@link UpdateScheduleRegistry} class.
 * @author ej-technologies GmbH
 */
public enum UpdateSchedule {

    /**
     *  {@link UpdateScheduleRegistry#checkAndReset()} always returns true
     */
    ON_EVERY_START("On every start", "start"),

    /**
     *  {@link UpdateScheduleRegistry#checkAndReset()} always returns true after more than one day has passed since
     * the last update
     */
    DAILY("Daily", "daily"),

    /**
     *  {@link UpdateScheduleRegistry#checkAndReset()} always returns true after more than one week has passed since
     * the last update
     */
    WEEKLY("Weekly", "weekly"),

    /**
     *  {@link UpdateScheduleRegistry#checkAndReset()} always returns true after more than one month has passed since
     * the last update
     */
    MONTHLY("Monthly", "monthly"),

    /**
     *  {@link UpdateScheduleRegistry#checkAndReset()} always returns false
     */
    NEVER("Never", "never");

    /**
     * All values of {@code UpdateSchedule} for use in a drop-down list.
     * <p>Not needed since this class became an enum in install4j 6.
     * @deprecated
     */
    @Deprecated public static final UpdateSchedule[] ALL_VALUES = values();

    /**
     * Returns the {@code UpdateSchedule} for a string ID. String IDs for  {@code UpdateSchedule}s are returned by
     * {@link #getId()}}.
     * @param id the ID
     * @return the {@code UpdateSchedule} or null if the ID is invalid
     */
    public static UpdateSchedule getById(String id) {
        for (UpdateSchedule updateSchedule : UpdateSchedule.values()) {
            if (Objects.equals(updateSchedule.getId(), id)) {
                return updateSchedule;
            }
        }
        return null;
    }

    private final String verbose;
    private final String id;

    UpdateSchedule(String verbose, String id) {
        this.verbose = verbose;
        this.id = id;
    }

    /**
     * Returns the string ID for the {@code UpdateSchedule}. This string value can be saved and restored later with the
     * {@link #getById(String)}} method.
     * @return the ID
     */
    public String getId() {
        return id;
    }

    @Override
    public String toString() {
        return verbose;
    }

}
