package com.install4j.api.screens;

import com.install4j.api.beans.VisualContainerBean;
import com.install4j.api.context.ProgressInterface;
import com.install4j.api.context.UserCanceledException;
import com.install4j.api.formcomponents.FormEnvironment;
import com.install4j.api.formcomponents.WidgetStyleHandler;

import javax.swing.*;

/**
 * The base interface for all screens. Do not implement this interface directly, only the derived interfaces are
 * functional. It is recommended that you choose one of the derived abstract classes as
 * a superclass. The life-cycle of screens is controlled by the framework.

 * @author ej-technologies GmbH
 */
public interface Screen extends VisualContainerBean {

    /**
     * Return the visual component that is displayed for the screen. This method is only called once, so the
     * result does not have to be cached.
     *
     * <p>In console or unattended mode, this method is never called.</p>
     * @return the component
     */
    JComponent createComponent();

    /**
     * Returns the title of this custom screen which is displayed at the top of the
     * installer wizard.
     *
     * <p>In console or unattended mode, this method is never called.</p>
     * @return the title.
     */
    String getTitle();

    /**
     * Returns the subtitle of this custom screen which is displayed at the top of the
     * installer wizard just below the title of the screen.
     *
     * <p>In unattended mode, this method is never called.
     * In console mode, the result of this method will be displayed for each screen.</p>
     * @return the title.
     */
    String getSubTitle();

    /**
     * Returns whether the "Next" button is visible or not for this screen. If you just want to disable the
     * button initially, please override {@code activate} and invoke {@code setNextButtonEnabled} in the
     * {@code WizardContext} that is available from the {@code Context} in GUI mode.
     *
     * <p>In console or unattended mode, this method is never called.</p>
     * @return {@code true} or {@code false}. Default is {@code true}.
     * @see com.install4j.api.context.WizardContext
     */
    boolean isNextVisible();

    /**
     * Returns whether the "Back" button is visible or not for this screen. If you just want to disable the
     * button initially, please override {@code activate} and invoke {@code setPreviousButtonEnabled} in the
     * {@code WizardContext} that is available from the {@code Context} in GUI mode.
     *
     * <p>In console or unattended mode, this method is never called.</p>
     * @return {@code true} or {@code false}. Default is {@code true}.
     * @see com.install4j.api.context.WizardContext
     */
    boolean isPreviousVisible();

    /**
     * Returns whether the "Cancel" button is visible or not for this screen. If you just want to disable the
     * button initially, please override {@code activate} and invoke {@code setCancelButtonEnabled} in the
     * {@code WizardContext} that is available from the {@code Context} in GUI mode.
     *
     * <p>In console or unattended mode, this method is never called.</p>
     * @return {@code true} or {@code false}. Default is {@code true}.
     * @see com.install4j.api.context.WizardContext
     */
    boolean isCancelVisible();

    /**
     * Called by the framework just before the screen is activated. Override this method to perform any
     * special initialization.
     *
     * <p>In console or unattended mode, this method is never called.</p>
     */
    void willActivate();

    /**
     * Called by the framework just after the screen has been activated. Override this method to perform any
     * special initialization.
     *
     * <p>In console or unattended mode, this method is never called.</p>
     */
    void activated();

    /**
     * Called by the framework just after the screen has been deactivated. Override this method to perform any
     * special cleanup.
     *
     * <p>In console or unattended mode, this method is never called.</p>
     */
    void deactivated();

    /**
     * Called when the user clicks the "Next" button for this screen. You can veto the change to the next
     * screen if you return {@code false}.
     *
     * <p>In console or unattended mode, this method is never called.</p>
     * @return whether the screen change is accepted or not.
     */
    boolean next();

    /**
     * Called when the user clicks the "Back" button for this screen. You can veto the change to the previous
     * screen if you return {@code false}.
     *
     * <p>In console or unattended mode, this method is never called.</p>
     * @return whether the screen change is accepted or not.
     */
    boolean previous();

    /**
     * Called when the user clicks the "Cancel" button for this screen. You can veto the cancel action
     * if you return {@code false}.
     * @return whether the cancellation is accepted or not.
     *
     * <p>In console or unattended mode, this method is never called.</p>
     */
    boolean cancel();

    /**
     * Returns whether this screen should be hidden when the user traverses screens
     * in the forward direction.
     *
     * <p>This method is also called in console or unattended mode.</p>
     * @return {@code true} or {@code false}.
     */
    boolean isHiddenForNext();

    /**
     * Returns whether this screen should be hidden when the user traverses screens
     * in the backward direction.
     *
     * <p>This method is also called in console or unattended mode.</p>
     * @return {@code true} or {@code false}.
     */
    boolean isHiddenForPrevious();

    /**
     * Returns whether this screen should be hidden. If this method returns {@code true},
     * it overrides the results of {@code isHiddenForNext} and {@code isHiddenForPrevious}.
     *
     * <p>This method is also called in console or unattended mode.</p>
     * @return {@code true} or {@code false}.
     * @see #isHiddenForNext()
     * @see #isHiddenForPrevious()
     */
    boolean isHidden();

    /**
     * Handle the unattended mode. This method is called when the screen is traversed in unattended mode.
     * There is no way to interact with the user. This method might be necessary to
     * mirror some behavior from the GUI mode, such as setting installer variables or configuring actions.
     * @return whether the installer or uninstaller can proceed with the next screen or whether the process should be canceled.
     */
    boolean handleUnattended();

    /**
     * Handle the console mode. This method is called when the screen is entered in console mode.
     * You can use the {@code Console} object to interact with the user and replicate the GUI functionality
     * on the terminal.
     * <p>If this screen has a form panel (i.e. {@code hasFormPanel()} returns {@code true}), you should call
     * {@code FormEnvironment.handleConsole} in this method to handle console mode for the contained
     * form components.
     * @param console the {@code Console} object
     * @return whether the installer or uninstaller can proceed with the next screen or whether the process should be canceled.
     * @throws UserCanceledException if the user cancels a question or notice. These exceptions are thrown by methods in the {@code Console} object.
     * @see FormEnvironment#handleConsole(Console)
     */
    boolean handleConsole(Console console) throws UserCanceledException;

    /**
     * Replace the default progress interface for actions with a custom progress interface.
     * When associated actions are run for a screen, a default progress interface is passed to them. If your screen has the possibility to
     * show progress information, you can return a different progress interface here. Typically you would only implement
     * methods regarding status and detail messages as well as methods that apply to a progress bar, other methods can be
     * delegated to the default progress interface. This is the strategy employed by the "Installation screen" and the
     * customizable "Progress screens".
     * @param defaultProgressInterface the default progress interface
     * @return the replaced progress interface
     */
    ProgressInterface getProgressInterface(ProgressInterface defaultProgressInterface);

    /**
     * This method is called by the framework to determine if the screen should only be created when it is shown.
     * By default, all screens provided by install4 are <em>not</em> created lazily. If you depend on custom code, that
     * has to be installed before {@link #createComponent()} can be called, then you should return {@code true} in
     * this method.
     * <p>If you have many screens that are created lazily and want to initialize them all at once, maybe after the
     * "Install files" action, you can call {@link com.install4j.api.context.Context#initializeLazilyCreatedScreens()}.
     * @return {@code true} or {@code false}.
     */
    boolean isCreateLazily();

    /**
     * This method is called by the framework to set the  {@code WidgetStyleHandler} just after the
     * screen has been constructed.
     * @param widgetStyleHandler the widget style handler.
     */
    void setWidgetStyleHandler(WidgetStyleHandler widgetStyleHandler);

}
