package com.install4j.api.events;

import com.install4j.api.context.Context;

import java.util.EventObject;

/**
 * The event object that is passed to installer event listeners.
 * Event listeners are registered in the {@link com.install4j.api.context.InstallerContext}.
 * @author ej-technologies GmbH
 * @see InstallerEventListener
 * @see com.install4j.api.context.InstallerContext#addInstallerEventListener(InstallerEventListener)
 */
public class InstallerEvent extends EventObject {

    private transient Context context;
    private EventType type;

    private Boolean success;

    /**
     * The constructor is called by the framework.
     * @param source the event source
     * @param context the context
     * @param type the event type.
     * @param success whether the operation associated with the event was successful
     */
    public InstallerEvent(Object source, Context context, EventType type, boolean success) {
        this(source, context, type);
        this.success = success;
    }

    /**
     * The constructor is called by the framework.
     * @param source the event source
     * @param context the context
     * @param type the event type.
     */
    public InstallerEvent(Object source, Context context, EventType type) {
        super(source);
        this.context = context;
        this.type = type;
    }

    /**
     * Get the type of the event.
     * @return the type
     */
    public EventType getType() {
        return type;
    }

    /**
     * Get the context.
     * @return the context.
     */
    public Context getContext() {
        return context;
    }

    public void setContext(Context context) {
        this.context = context;
    }

    /**
     * Return whether the operation associated with the event was successful
     * @return {@code true} or {@code false}.
     */
    public boolean wasSuccessful() {
        return success == null || success;
    }

    @Override
    public String toString() {
        return super.toString() + " [type " + type + "]";
    }

    /**
     * Return a verbose description of this event.
     * @return the description
     */
    public String getVerbose() {
        if (wasSuccessful()) {
            if (success == null) {
                return type.getOperationVerbose();
            } else {
                return type.getOperationVerbose() + " successful";
            }
        } else {
            return type.getOperationVerbose() + " not successful";
        }
    }
}
