package com.install4j.api.events;

/**
 * Enumeration class that represents all different event types for instances of {@link InstallerEvent}.
 * Installer events cover everything that is written to the installer log file as well
 * as application life-cycle events.
 * @author ej-technologies GmbH
 */
public enum EventType {

    /**
     * Fired before an action is executed
     */
    BEFORE_EXECUTE_ACTION("Before execute action", "Execute action"),

    /**
     * Fired after an action has been executed. The event object will be of type {@link InstallerActionEvent}.
     */
    AFTER_EXECUTE_ACTION("After execute action", "Execute action"),

    /**
     * Fired before a rollback is performed
     */
    BEFORE_ROLLBACK_ACTION("Before rollback action", "Rollback action"),

    /**
     * Fired after a rollback is performed
     */
    AFTER_ROLLBACK_ACTION("After rollback action", "Rollback action finished"),

    /**
     * Fired before a file is installed. The event object will be of type {@link InstallerFileEvent}.
     */
    BEFORE_INSTALL_FILE("Before install file", "Install file"),

    /**
     * Fired after a file has been installed. The event object will be of type {@link InstallerFileEvent}.
     */
    AFTER_INSTALL_FILE("After install file", "Install file"),

    /**
     * Fired after a directory has been installed. The event object will be of type {@link InstallerDirectoryEvent}.
     */
    DIRECTORY_CREATED("After create directory", "Created directory"),

    /**
     * Fired before a file is downloaded. The event object will be of type {@link InstallerDownloadEvent}.
     */
    BEFORE_DOWNLOAD("Before a download", "Download"),

    /**
     * Fired before a file has been downloaded. The event object will be of type {@link InstallerDownloadEvent}.
     */
    AFTER_DOWNLOAD("After a download", "Download"),

    /**
     * Fired before a screen is shown.
     */
    SHOW_SCREEN("Show screen"),

    /**
     * Fired when the installer or uninstaller quits but before any cleanup has been performed.
     */
    FINISHING("Finishing"),

    /**
     * Fired just before the installer or uninstaller quits.
     */
    FINISHED("Finished"),

    /**
     * Fired when the installer or uninstaller is canceled but before any cleanup has been performed.
     */
    CANCELLING("Cancelling"),

    /**
     * Fired just before the installer or uninstaller quits after having been canceled.
     */
    CANCELED("Canceled"),

    /**
     * Fired after a variable has been set.
     * @see com.install4j.api.context.Context#setVariable(String, Object)
     */
    VARIABLE_CHANGED("Variable changed"),

    /**
     * Fired before the file installation is started by the "Install files" action. You can get the total
     * size and the total file count from the event object.
     *
     */
    FILE_INSTALLATION_STARTED("File installation started");

    private final String eventVerbose;
    private final String operationVerbose;

    EventType(String eventVerbose) {
        this(eventVerbose, eventVerbose);
    }

    EventType(String eventVerbose, String operationVerbose) {
        this.operationVerbose = operationVerbose;
        this.eventVerbose = eventVerbose;
    }

    /**
     * Get a verbose message that describes the operation that is connected with this event.
     * @return the message
     */
    public String getOperationVerbose() {
        return operationVerbose;
    }

    @Override
    public String toString() {
        return eventVerbose;
    }
}
