package com.install4j.api.context;

import com.install4j.api.ApplicationRegistry;

import java.io.File;

/**
 * The context provided by the installer.
 * In addition to the methods in the {@code Context} interface, this extended context offers methods that are specific
 * to the installer.
 * @author ej-technologies GmbH
 */
public interface InstallerContext extends Context {


    /**
     * Set the installation directory programmatically. Based on the configuration in the media wizard, there is a default
     * installation directory. The "Installation directory" screen will suggest the location returned by
     * {@link Context#getInstallationDirectory()}
     * and subsequently call {@code setInstallationDirectory} with the user selection. To change the default installation directory,
     * you could call this method from a "Run script" action that is added to the "Startup" screen.
     * @param installationDirectory the new installation directory
     */
    void setInstallationDirectory(File installationDirectory);

    /**
     * Returns whether the installer is or will be updating an existing version of the same program.
     * This method returns {@code true} if {@link ApplicationRegistry#isUpdateDirectory(File)}
     * returns {@code true} for the currently set installation directory ({@link #getInstallationDirectory()}.
     * @return {@code true} or {@code false}.
     */
    boolean isUpdateInstallation();

    /**
     * Returns the current media file. On macOS it returns the base directory of the installer app bundle.
     * @return the current media file. Returns {@code null} if you debug the installer, since there is no installer file
     * in that case.
     */
    File getInstallerFile();

    /**
     * Get the name of the current media file.
     * @return the original file name of the media file being installed without the file extension.
     */
    String getMediaName();

    /**
     * Installs a file. This is the same as calling {@link #installFile(File, File, FileOptions)}
     * with {@link FileOptions#FileOptions(long) new FileOptions(sourceFile.lastModified())}.
     * The uninstaller will uninstall a file that has been installed with this method. The overwrite mode
     * will be {@code OverwriteMode.ALWAYS_ASK_EXCEPT_FOR_UPDATE} and the uninstaller mode
     * {@code UninstallMode.IF_CREATED}.
     * @param sourceFile the file to be installed.
     * @param destFile the destination file.
     * @return whether the installation was successful or not
     * @throws UserCanceledException if the user cancels the installation.
     */
    boolean installFile(File sourceFile, File destFile) throws UserCanceledException;

    /**
     * Installs a file. This is the same as calling {@link #installFile(File, File, FileOptions, ProgressInterface, int, int)}
     * with a {@code ProgressInterface} of {@code null}. The
     * {@code FileOptions} parameter determines the overwrite and uninstall modes.
     * @param sourceFile the file to be installed.
     * @param destFile the destination file.
     * @param options the installation and uninstallation options.
     * @return whether the installation was successful or not
     * @throws UserCanceledException if the user cancels the installation.
     */
    boolean installFile(File sourceFile, File destFile, FileOptions options) throws UserCanceledException;

    /**
     * Installs a file. The method will try to copy the source file to the destination file. The
     * {@code FileOptions} parameter determines the overwrite mode and the uninstall mode.
     * With this method, the progress interface will be updated during the installation.
     * Use this method if you install very large files.
     * <p>Note that the progress bar is only available if this method is called from an action and the action is associated
     * with a screen that offers a progress interface with a progress bar, such as the "Installation" screen or the
     * customizable "Display progress" screen.
     * @param sourceFile the file to be installed.
     * @param destFile the destination file.
     * @param options the installation and uninstallation options.
     * @param progressInterface the interface for displaying progress during the file installation.
     *  The current progress interface can be obtained from {@link Context#getProgressInterface()}.
     * @param fromPercent the start percentage value for the file installation. The progress interface
     *                    will be advanced starting with this percentage value.
     * @param toPercent the end percentage value for the file installation. The progress interface
   *                    will be advanced up to this percentage value.
     * @return whether the installation was successful or not
     * @throws UserCanceledException if the user cancels the installation.
     */
    boolean installFile(File sourceFile, File destFile, FileOptions options, ProgressInterface progressInterface, int fromPercent, int toPercent) throws UserCanceledException;


    /**
     * Creates a directory. The directory will be deleted by the uninstaller if all its containing files and directories are also uninstalled
     * and the {@code UninstallMode} requires an uninstallation. This method is only necessary if the directory will be empty and you do not know if the directory will
     * contain files. If you install a known set of files, you can just use the {@code installFile(...)} methods without creating the directory first.
     * @param directory the directory to be created. All parent directories will be created if necessary.
     * @param uninstallMode the behavior for uninstallation. It is recommended to use {@link UninstallMode#IF_CREATED} by default.
     */
    void installDirectory(File directory, UninstallMode uninstallMode);

    /**
     * Registers an arbitrary file for uninstallation.
     * @param file the file that should be removed when uninstalling the application.
     */
    void registerUninstallFile(File file);

}
