package com.install4j.api.beans;

import java.io.Serializable;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;

/**
 * Container for a set of localized files that represent a single {@link ExternalFile} with textual content.
 * If you want the user to be able to specify localized versions of a text file for a single property,
 * you can declare the property to be of type {@code LocalizedExternalFile}. Important:
 * you have to initialize the field with an instance rather than leave it to be {@code null}.
 * <p>In the install4j IDE, the user will be presented with a dialog that allows file selection for all
 * installer languages that are configured on the General Settings-&gt;Languages tab. At runtime, call
 * {@link com.install4j.api.context.Context#getExternalFile(LocalizedExternalFile, boolean)} to
 * get the actual file for further use in your bean implementation.
 * <p>In your bean info, use the {@link com.install4j.api.beaninfo.FilePropertyDescriptor} class to
 * register the property. All attributes of {@code FilePropertyDescriptor} are supported
 * for {@code LocalizedExternalFile}.
 *
 * @see ExternalFile
 * @author ej-technologies GmbH
 */
public class LocalizedExternalFile implements Serializable {

    private Map<String, ExternalFile> languageIdToExternalFile = new LinkedHashMap<>();

    /**
     * Returns the map from language IDs to {@link ExternalFile}s. You do not
     * have to use this method, a more convenient way to retrieve the actual file at
     * runtime is to call {@link com.install4j.api.context.Context#getExternalFile(LocalizedExternalFile, boolean)}.
     * @return the map
     */
    public Map<String, ExternalFile> getLanguageIdToExternalFile() {
        return languageIdToExternalFile;
    }

    /**
     * Sets the map from language IDs to {@link ExternalFile}s. This method is called by
     * the install4j IDE.
     * @param languageIdToExternalFile the map
     */
    public void setLanguageIdToExternalFile(Map<String, ExternalFile> languageIdToExternalFile) {
        this.languageIdToExternalFile = languageIdToExternalFile;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        LocalizedExternalFile that = (LocalizedExternalFile)o;
        return Objects.equals(languageIdToExternalFile, that.languageIdToExternalFile);
    }

    @Override
    public int hashCode() {
        return Objects.hash(languageIdToExternalFile);
    }
}
