package com.install4j.api.beaninfo;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;

/**
 * Descriptor for script parameters of script properties defined with a {@link ScriptPropertyDescriptor}.
 * @see com.install4j.api.beans.ScriptProperty
 */
public class ScriptParameter {

    private String parameterName;
    private Type parameterType;

    /**
     * Create a script parameter.
     * @param parameterName the parameter name
     * @param parameterClass the parameter class
     */
    public ScriptParameter(String parameterName, Class parameterClass) {
        this.parameterName = parameterName;
        this.parameterType = parameterClass;
    }

    /**
     * Create a script parameter with a generic type.
     * Getting a concrete generic type requires some work, for example, to get a type for {@code List&lt;String&gt;}, you
     * define a helper interface
     * <pre>{@code
     *   private interface StringList extends List&lt;String[]&gt; {}
     * }</pre>
     * and query the generic super-interface to get the desired generic type:
     * <pre>{@code
     *   (ParameterizedType)StringArrayList.class.getGenericInterfaces()[0]
     * }</pre>
     * @param parameterName the parameter name
     * @param parameterizedType the parameter type
     */
    public ScriptParameter(String parameterName, ParameterizedType parameterizedType) {
        this.parameterName = parameterName;
        this.parameterType = parameterizedType;
    }

    /**
     * Get the parameter name
     * @return the name
     */
    public String getParameterName() {
        return parameterName;
    }

    /**
     * Get the parameter class.
     * If the script parameter has been created with a parameterized type, this returns the raw type.
     * @return the class
     */
    public Class getParameterClass() {
        if (parameterType == null) {
            return null;
        } else if (parameterType instanceof Class) {
            return (Class)parameterType;
        } else if (parameterType instanceof ParameterizedType) {
            return (Class)((ParameterizedType)parameterType).getRawType();
        } else {
            throw new IllegalStateException(parameterType.getClass().getName());
        }
    }

    /**
     * Get the parameter type
     * @return the type
     */
    public Type getParameterType() {
        return parameterType;
    }
}
