package com.install4j.api.windows;

import com.install4j.api.Util;
import com.install4j.api.context.UserCanceledException;
import com.install4j.runtime.installer.helper.MenuHelper;
import com.install4j.runtime.installer.helper.comm.ExecutionContext;
import com.install4j.runtime.installer.platform.win32.FolderInfo;
import com.install4j.runtime.installer.platform.win32.Misc;
import com.install4j.runtime.installer.platform.win32.ShellLink;

import java.io.File;
import java.io.IOException;

/**
 * Collection of static methods to access Microsoft Windows-specific
 * folder locations and to execute native file system operations.
 * @author ej-technologies GmbH
 */
public class WinFileSystem {

    /**
     * Returns Windows-specific directories like the start menu or the
     * desktop.
     * @param folderType specifies the requested folder type.
     * @param allUsers if this parameter is {@code true} and if the current user is in the
     * Administrators group, the method returns the requested folder for all users.
     * Otherwise, it returns the folder for the current user.
     * @return the requested folder location.
     */
    public static File getSpecialFolder(SpecialFolder folderType, boolean allUsers) {
        return FolderInfo.getSpecialFolder(folderType.getIntValue(), allUsers);
    }

    /**
     * Returns the program files directory of the current Windows installation.
     * A typical path is {@code C:\Program Files}.
     * @return the program files directory.
     */
    public static File getProgramFilesDirectory() {
        return FolderInfo.getProgramFilesDirectory();
    }

    /**
     * Returns a folder for components that are shared across applications.
     * A typical path is {@code C:\Program Files\Common Files}.
     * @return the common files directory.
     */
    public static File getCommonFilesDirectory() {
        return FolderInfo.getCommonFilesDirectory();
    }

    /**
     * Returns the program data directory where applications can save data that is not specific to particular users.
     * A typical path is {@code C:\ProgramData}.
     * @return the common files directory.
     */
    public static File getProgramDataDirectory() {
        return FolderInfo.getProgramDataDirectory();
    }

    /**
     * Returns the Windows directory.
     * A typical path is {@code C:\Windows}.
     * @return the Windows directory.
     */
    public static File getWindowsDirectory() {
        return FolderInfo.getWindowsDirectory();
    }

    /**
     * Returns the system directory.
     * A typical path is {@code C:\Windows\system32}
     * @return the system directory.
     */
    public static File getSystemDirectory() {
        return FolderInfo.getSystemDirectory();
    }

    /**
     * Creates a shell link.
     * Note that Windows might change the link suffix from {@code lnk} to {@code pif},
     * if you are linking to a DOS executable or a batch file.
     * This is the same as calling {@code createShellLink(file, shortcutTo, iconFile, null, null)}
     * @param file the shell link that should be created including the {@code lnk} suffix.
     * @param shortcutTo the file it should link to.
     * @param iconFile an optional icon file. If {@code null} is passed, the default icon of the
     *                 {@code shortcutTo} target will be used.
     * @return whether the link could be created or not.
     * @see #createShellLink(File, File, File, String)
     */
    public static boolean createShellLink(File file, File shortcutTo, File iconFile) {
        return createShellLink(file, shortcutTo, iconFile, null, null);
    }

    /**
     * Creates a shell link.
     * Note that Windows might change the link suffix from {@code lnk} to {@code pif},
     * if you are linking to a DOS executable or a batch file.
     * This is the same as calling {@code createShellLink(file, shortcutTo, iconFile, arguments, null)}
     * @param file the shell link that should be created including the {@code lnk} suffix.
     * @param shortcutTo the file it should link to.
     * @param iconFile an optional icon file. If {@code null} is passed, the default icon of the
     *                 {@code shortcutTo} target will be used.
     * @param arguments the arguments passed to the target.
     * @return whether the link could be created or not.
     */
    public static boolean createShellLink(File file, File shortcutTo, File iconFile, String arguments) {
        return createShellLink(file, shortcutTo, iconFile, arguments, null);
    }

    /**
     * Creates a shell link.
     * Note that Windows might change the link suffix from {@code lnk} to {@code pif},
     * if you are linking to a DOS executable or a batch file.
     *
     * @param file the shell link that should be created including the {@code lnk} suffix.
     * @param shortcutTo the file it should link to.
     * @param iconFile an optional icon file. If {@code null} is passed, the default icon of the
     *                 {@code shortcutTo} target will be used.
     * @param arguments the arguments passed to the target.
     * @return whether the link could be created or not.
     * @param description the description used in the tooltip
     */
    public static boolean createShellLink(File file, File shortcutTo, File iconFile, String arguments, String description) {
        return createShellLink(file, shortcutTo, iconFile, arguments, description, null);
    }

    /**
     * Creates a shell link.
     * Note that Windows might change the link suffix from {@code lnk} to {@code pif},
     * if you are linking to a DOS executable or a batch file.
     *
     * @param file the shell link that should be created including the {@code lnk} suffix.
     * @param shortcutTo the file it should link to.
     * @param iconFile an optional icon file. If {@code null} is passed, the default icon of the
     *                 {@code shortcutTo} target will be used.
     * @param arguments the arguments passed to the target.
     * @return whether the link could be created or not.
     * @param description the description used in the tooltip
     * @param startIn the working directory for the executable
     */
    public static boolean createShellLink(File file, File shortcutTo, File iconFile, String arguments, String description, File startIn) {
        return createShellLink(file, shortcutTo, iconFile, arguments, description, startIn, ShowCommand.NORMAL);
    }

    /**
     * Creates a shell link.
     * Note that Windows might change the link suffix from {@code lnk} to {@code pif},
     * if you are linking to a DOS executable or a batch file.
     *
     * @param file the shell link that should be created including the {@code lnk} suffix.
     * @param shortcutTo the file it should link to.
     * @param iconFile an optional icon file. If {@code null} is passed, the default icon of the
     *                 {@code shortcutTo} target will be used.
     * @param arguments the arguments passed to the target.
     * @return whether the link could be created or not.
     * @param description the description used in the tooltip
     * @param startIn the working directory for the executable
     * @param showCommand the window show command
     */
    public static boolean createShellLink(File file, File shortcutTo, File iconFile, String arguments, String description, File startIn, ShowCommand showCommand) {
        return ShellLink.create(file, shortcutTo, iconFile, arguments, false, description, startIn, getShowCommandInt(showCommand));
    }

    private static int getShowCommandInt(ShowCommand showCommand) {
        if (showCommand != null) {
            switch (showCommand) {
                case MAXIMIZED:
                    return ShellLink.SW_SHOWMAXIMIZED;
                case MINIMIZED:
                    return ShellLink.SW_SHOWMINNOACTIVE;
            }
        }
        return ShellLink.SW_SHOWNORMAL;
    }


    /**
     * Creates a start menu entry.
     * @param folder the name of the folder where the entry should be created. Leave empty for top-level entries.
     * @param entryName the name of the menu entry. You can create subfolders with backslashes.
     * @param destination the file that should be linked to
     * @param allUsers whether the entry should be created for all users or not
     * @param icon an optional icon file. If {@code null} is passed, the default icon of the
     *                 {@code destination} target will be used.
     * @throws IOException if the start menu entry could not be created
     */

    public static void createStartMenuEntry(String folder, String entryName, File destination, boolean allUsers, File icon) throws IOException {
        createStartMenuEntry(folder, entryName, destination, allUsers, icon, null);
    }

    /**
     * Creates a start menu entry.
     * @param folder the name of the folder where the entry should be created. Leave empty for top-level entries.
     * @param entryName the name of the menu entry. You can create subfolders with backslashes.
     * @param destination the file that should be linked to
     * @param allUsers whether the entry should be created for all users or not
     * @param icon an optional icon file. If {@code null} is passed, the default icon of the
     *                 {@code destination} target will be used.
     * @param arguments optional arguments given to the destination. Can be {@code null}
     * @throws IOException if the start menu entry could not be created
     */

    public static void createStartMenuEntry(String folder, String entryName, File destination, boolean allUsers, File icon, String arguments) throws IOException {
        createStartMenuEntry(folder, entryName, destination, allUsers, icon, arguments, false);
    }

    /**
     * Creates a start menu entry.
     *
     * @param folder the name of the folder where the entry should be created. Leave empty for top-level entries.
     * @param entryName the name of the menu entry. You can create subfolders with backslashes.
     * @param destination the file that should be linked to
     * @param allUsers whether the entry should be created for all users or not
     * @param icon an optional icon file. If {@code null} is passed, the default icon of the
     *                 {@code destination} target will be used.
     * @param arguments optional arguments given to the destination. Can be {@code null}
     * @param runAsAdministrator whether the entry should be run as administrator
     * @throws IOException if the start menu entry could not be created
     */

    public static void createStartMenuEntry(String folder, String entryName, File destination,
                                            boolean allUsers, File icon, String arguments, boolean runAsAdministrator) throws IOException {
        File groupDir = new File(FolderInfo.getSpecialFolder(FolderInfo.FOLDER_PROGRAMS, allUsers), folder);
        File menuFile = new File(groupDir, entryName);
        try {
            MenuHelper.installWindowsMenu(allUsers ? ExecutionContext.MAXIMUM : ExecutionContext.UNELEVATED, menuFile, destination,
                    icon, arguments, runAsAdministrator, null);
        } catch (UserCanceledException e) {
            e.printStackTrace();
            throw new IOException(e.toString());
        }
    }

    /**
     * Calls the win32 API function MoveFileEx with MOVEFILE_DELAY_UNTIL_REBOOT.
     * You must be an administrator to call this function.
     *
     * @param source the source file. Must be on the same drive as the destination file.
     * @param destination the destination file. If empty, the source file will be deleted.
     * @return true if the operation could be registered successfully.
     */
    public static boolean moveWithDelayUntilReboot(File source, File destination) {
        return Misc.moveWithDelayUntilReboot(source, destination);
    }


    /**
     * Tests on what kind of drive the given file is (or would be) located.
     * @param file an existing or non-existing file
     * @return a constant of {@link DriveType}.
     */
    public static DriveType getDriveType(File file) {
        if (Util.isWindows()) {
            String path = file.getAbsolutePath();
            if (path.length() >= 3 && !path.startsWith("\\")) {
                return DriveType.getFromIntValue(FolderInfo.getDriveType(path.substring(0, 3)));
            }
        }
        return DriveType.UNKNOWN;
    }


    private WinFileSystem() {
    }
}
