package com.install4j.api.launcher;

import com.exe4j.runtime.util.FileUtil;
import com.exe4j.runtime.util.ResourceHelper;
import com.install4j.api.Util;
import com.install4j.runtime.installer.InstallerConstants;
import com.install4j.runtime.installer.InstallerContextImpl;
import com.install4j.runtime.installer.InstallerVariables;
import com.install4j.runtime.installer.config.InstallerConfig;
import com.install4j.runtime.installer.helper.InstallerUtil;
import com.install4j.runtime.installer.helper.apiimpl.VariablesImpl;
import com.install4j.runtime.launcher.LauncherVariables;

import java.io.File;
import java.io.IOException;
import java.util.Map;

/**
 * This class provides methods to access compiler and installer variables from your launchers.
 * Do not use this class in the installer, uninstaller or in a custom installer application, in that case use
 * the methods in the {@link com.install4j.api.context.Context context} instead.
 */
public class Variables {

    private static Map<String, String> compilerVariables;
    private static Map<String, Object> installerVariables;

    private Variables() {
    }

    /**
     * Get the value of a compiler variable. The first time this method is called, the installer config file will be read.
     * For subsequent calls the compiler variables will be cached.
     * @param variableName the name of the compiler variable
     * @return the value of the compiler variable or {@code null} if the variable is undefined
     * @throws IOException if the installer config file cannot be read
     */
    public static String getCompilerVariable(String variableName) throws IOException {
        if (compilerVariables == null) {
            InstallerConfig config = InstallerConfig.getConfigFromFile(InstallerUtil.getInstallerFile(InstallerConstants.CONFFILE_NAME));
            compilerVariables = config.getCompilerVariables();
        }
        return compilerVariables.get(variableName);
    }

    /**
     * Get a map of installer variables that were saved to the automatically created response file {@code response.varfile}.
     * Not all installer variables are saved to that file. You can register a variable as a response file variable
     * with {@link com.install4j.api.context.Context#registerResponseFileVariable(String)}. Variables to which
     * form components are bound are automatically registered as response file variables.
     * <p>If the automatically created response file cannot be read, only system installer variables are available.
     * @return the map with installer variables from the response file. The keys are the variable names, the values are
     * the original values at the time that the response file was saved.
     */
    public synchronized static Map<String, Object> getInstallerVariables() {
        ensureInstallerVariablesRead();
        return installerVariables;
    }

    /**
     * Get the value of an installer variable that was saved to the automatically created response file {@code response.varfile}.
     * See {@link #getInstallerVariables()} for more information.
     * <p>If the automatically created response file cannot be read, only system installer variables are available.
     * @param variableName the name of the compiler variable
     * @return the value of the compiler variable or {@code null} if the variable is undefined
     * @see #getInstallerVariables()
     */
    public static synchronized Object getInstallerVariable(String variableName) {
        ensureInstallerVariablesRead();
        return installerVariables.get(variableName);
    }

    /**
     * Load installer variables from the preference stores that have been saved by a "Save installer variables to the preference store" action.
     * The package name in the preference store is set to the application ID, which is the default save location of that action.
     * @param userSpecific if the user-specific preference store should be used or not
     * @return the map with installer variables from the preference store. The keys are the variable names, the values are
     * the original values at the time that the installer variables were saved. Returns {@code null} if no
     * variables were saved.
     * @throws IOException if the operation fails due to a problem with the backing store
     */
    public static Map<String, Object> loadFromPreferenceStore(boolean userSpecific) throws IOException {
        return loadFromPreferenceStore(LauncherVariables.getApplicationId(), userSpecific);
    }

    /**
     * Load installer variables from the preference stores that have been saved by a "Save installer variables to the preference store" action.
     * @param packageName the package name that was used when saving the installer variables to the preference store
     * @param userSpecific if the user-specific preference store should be used or not
     * @return the map with installer variables from the preference store. The keys are the variable names, the values are
     * the original values at the time that the installer variables were saved. Returns {@code null} if no
     * variables were saved.
     * @throws IOException if the operation fails due to a problem with the backing store
     */
    public static Map<String, Object> loadFromPreferenceStore(String packageName, boolean userSpecific) throws IOException {
        return VariablesImpl.loadVariablesFromPreferenceStore(packageName, userSpecific);
    }

    /**
     * Save a map of installer variables to the preference store. If you use installer variables loaded by
     * {@link #loadFromPreferenceStore(boolean)} for the configuration of your application,
     * you can save changes made in the application back to the preference store with this method. Values that cannot be
     * decoded as described by {@link com.install4j.api.context.Context#registerResponseFileVariable(String)} will be ignored.
     * Values that are present in the preference store and not in the supplied map will not be deleted.
     * The package name in the preference store is set to the application ID, which is the default save location of the
     * "Save installer variables to the preference store" action.
     * @param variables the map with the installer variables as returned by {@link #loadFromPreferenceStore(boolean)}
     * @param userSpecific if the user-specific preference store should be used or not
     * @throws IOException if the operation fails due to a problem with the backing store
     */
    public static void saveToPreferenceStore(Map<String, Object> variables, boolean userSpecific) throws IOException {
        saveToPreferenceStore(variables, LauncherVariables.getApplicationId(), userSpecific);
    }

    /**
     * Save a map of installer variables to the preference store. If you use installer variables loaded by
     * {@link #loadFromPreferenceStore(String, boolean)} for the configuration of your application,
     * you can save changes made in the application back to the preference store with this method. Values that cannot be
     * decoded as described by {@link com.install4j.api.context.Context#registerResponseFileVariable(String)} will be ignored.
     * Values that are present in the preference store and not in the supplied map will not be deleted.
     * @param variables the map with the installer variables as returned by {@link #loadFromPreferenceStore(String, boolean)}
     * @param packageName the package name to which the installer variables should be saved in the preference store
     * @param userSpecific if the user-specific preference store should be used or not
     * @throws IOException if the operation fails due to a problem with the backing store
     */
    public static void saveToPreferenceStore(Map<String, Object> variables, String packageName, boolean userSpecific) throws IOException {
        VariablesImpl.saveVariablesToPreferenceStore(variables, packageName, userSpecific);
    }

    /**
     * Clear the cache of read installer variables. If you know that the response file has changed, you can call this method
     * to force the installer variables to be read again. The methods in
     * {@link ApplicationLauncher} call this method automatically when the custom installer
     * application exits.
     */
    public synchronized static void clearInstallerVariablesCache() {
        installerVariables = null;
    }

    private synchronized static void ensureInstallerVariablesRead() {
        if (installerVariables == null) {
            installerVariables = VariablesImpl.readInstallerVariables(InstallerUtil.getInstallerFile(InstallerContextImpl.RESPONSE_FILE_NAME));
            addInstallationDir();
            if (System.getProperty(InstallerConstants.PROPNAME_COMM_IDENTIFIER) == null) { // Only if not in the elevated helper
                InstallerVariables.initUserVars(installerVariables);
                InstallerVariables.initSystemVars(installerVariables);
            }
        }
    }

    private static void addInstallationDir() {
        String contentDirectory = FileUtil.getCanonicalPath(new File(ResourceHelper.getRuntimeDir(), ".."));
        String installationDirectory = contentDirectory;

        if (Util.isMacOS() && (contentDirectory.endsWith(".app/Contents/java/app") || contentDirectory.endsWith(".app/Contents/Resources/app"))) {
            installationDirectory = FileUtil.getCanonicalPath(new File(contentDirectory, "../../../.."));
        }
        installerVariables.put(InstallerVariables.VARIABLE_INSTALLATION_DIR, installationDirectory);
        installerVariables.put(InstallerVariables.VARIABLE_CONTENT_DIR, contentDirectory);
    }
}
