package com.install4j.api;

import com.install4j.runtime.installer.helper.apiimpl.ApplicationRegistryImpl;

import java.io.File;

/**
 * The class provides information about applications installed with install4j.
 * @author ej-technologies GmbH
 */
public class ApplicationRegistry {

    /**
     * Checks if the application can be safely installed into the given directory.
     * If this is an add-on installer, it returns true if the directory contains the target app.
     * @param dir the directory to be checked
     * @return true if safe
     */
    public static boolean checkApplicationId(File dir) {
        return ApplicationRegistryImpl.checkApplicationId(dir);
    }

    /**
     * Checks if this application is already installed in the given directory.
     * This is the case if the application ID of the current installer is equal to the application
     * ID of the already installed application.
     * <p>Note that if you pass the installation
     * directory of the current installer to this method, the return value will change to
     * {@code true} after the "Install Files" action has run. To check if the current installer
     * will perform an update installation, use {@link com.install4j.api.context.InstallerContext#isUpdateInstallation()}
     * which returns a consistent value over the entire lifetime of the installer.
     * @param dir the directory to be checked
     * @return true if an update is detected
     * @see com.install4j.api.context.InstallerContext#isUpdateInstallation()
     */
    public static boolean isUpdateDirectory(File dir) {
        return ApplicationRegistryImpl.isUpdateDirectory(dir);
    }


    /**
     * Retrieves information about an application installed by install4j via its
     * application id.
     * @param id the application id
     * @return an array of ApplicationInfo objects containing all installations of
     * the application with this id. The newest installation is the first element.
     * Empty array if there are no installations.
     */
    public static ApplicationInfo[] getApplicationInfoById(String id) {
        return ApplicationRegistryImpl.getApplicationInfoById(id);
    }

    /**
     * Checks if the specified directory contains an application installed by install4j
     * and retrieves information about it.
     * @param dir the base directory to check
     * @return an ApplicationInfo object if the directory contains an application installed with install4j,
     * {@code null} otherwise.
     */
    public static ApplicationInfo getApplicationInfoByDir(File dir) {
        return ApplicationRegistryImpl.getApplicationInfoByDir(dir);
    }

    /**
     * Checks if the specified directory contains an application installed by install4j
     * and retrieves information about the add-on installers in it.
     * @param dir the base directory to check
     * @return an array of ApplicationInfo objects if the directory contains an application installed with install4j,
     * {@code null} otherwise. The array will be empty if there are no add-ons installed.
     */
    public static ApplicationInfo[] getAddOnApplicationInfoByDir(File dir) {
        return ApplicationRegistryImpl.getAddOnApplicationInfoByDir(dir);
    }


    /**
     * Provides information about an application installed by install4j.
     */
    public interface ApplicationInfo {
        /**
         * Returns the directory the application was installed in the last time the installer ran.
         * @return the installation directory
         */
        File getInstallationDirectory();

        /**
         * Returns the application ID.
         * @return the application id, {@code null} if the application was installed with install4j &lt; 3.0
         */
        String getId();

        /**
         * Returns the name of the application.
         * @return the full name of the application
         */
        String getName();

        /**
         * Returns the version of the application.
         * @return the version
         */
        String getVersion();


        /**
         * Returns the ID of the media file that was used to install this application.
         * @return the media file ID or an empty string if the application was installed with install4j &lt; 4.1
         */
        String getMediaSetId();

        /**
         * Get the value of a compiler variable in the application.
         * @param variableName the name of the compiler variable
         * @return the value of the compiler variable or {@code null} if the variable is undefined
         */
        String getCompilerVariable(String variableName);

        /**
         * Get the value of an installer variable that was saved to the automatically created response file {@code response.varfile}.
         * @param variableName the name of the compiler variable
         * @return the value of the compiler variable or {@code null} if the variable is undefined
         */
        Object getInstallerVariable(String variableName);

    }
}
