package com.install4j.api.styles;

import com.install4j.api.beans.VisualContainerBean;
import com.install4j.api.context.Context;
import com.install4j.api.screens.Screen;

import javax.swing.*;

/**
 * All styles must implement this interface. You have to override all methods and provide a public constructor with no arguments.
 * It is recommended that you choose {@link AbstractStyle} as as a superclass.
 * <p>
 * Styles are configurable in the install4j GUI on the "Installer-&gt;Styles" step.
 * The life-cycle of screen style instances is controlled by the framework. Styles are only instantiated
 * once and their {@link #createComponent()} method is called for each screen that the screen style is used for.
 * </p>
 * <p>
 * If you nest styles, implement {@link NestedStyleContainer} as well.
 * </p>
 *
 * @author ej-technologies GmbH
 */
public interface Style extends VisualContainerBean, StyleContextReceiver, ControlButtonContainer {

    /**
     * This method is called by the framework to set the  {@code Context} just after the
     * screen style has been constructed. This is either an {@code InstallerContext} or an
     * {@code UninstallerContext}, depending on whether the screen style is used in an
     * installer or an uninstaller.
     * <p>
     * When nesting other styles, this is the earliest occasion that the screen style manager can be obtained by calling
     * {@code context.getWizardContext().getStyleManager()}.
     * </p>
     *
     * @param context the context.
     */
    void setContext(Context context);

    /**
     * Create the UI component for the style. Depending on the return value of the {@link #isFillHorizontal()} and the
     * {@link #isFillVertical()} methods, the component grows to fill all available horizontal or vertical space.
     * The method must not return {@code null}.
     * <p>
     * Never call this method yourself. For nested styles, call {@link StyleManager#createStyleComponent(Style)}
     * instead.
     * </p>
     * <p>Each screen style is only instantiated once, and this method is called multiple times for each screen that uses
     * the style. In console or unattended mode, this method is never called.</p>
     *
     * @return the UI component
     * @see #isFillHorizontal()
     * @see #isFillVertical()
     * @see NestedStyleContainer
     */
    JComponent createComponent();

    /**
     * Returns if the screen style is intended to by applied to installer applications, screens and screen groups.
     * Only standalone styles are available in the corresponding screen style selectors and can be set as the default style.
     * If this method returns {@code false}, the screen style can only be nested into other styles.
     *
     * @return {@code true} or {@code false}
     */
    boolean isStandalone();

    /**
     * This method is called by the framework just before a screen with this screen style is activated.
     * This call has to be delegated to nested styles by using {@link StyleManager#willActivate(Style)}.
     *
     * @see Screen#willActivate()
     */
    void willActivate();

    /**
     * This method is called by the framework just after a screen with this screen style has been activated.
     * This call has to be delegated to nested styles by using {@link StyleManager#activated(Style)}
     *
     * @see Screen#activated()
     */
    void activated();

    /**
     * This method is called by the framework just after a screen with this screen style has been deactivated.
     * This call has to be delegated to nested styles by using {@link StyleManager#deactivated(Style)}
     */
    void deactivated();

}
