package com.install4j.api.launcher;

import com.install4j.runtime.installer.helper.applaunch.ApplicationLauncherImpl;
import com.install4j.runtime.launcher.integration.FirstRun;

import java.awt.*;
import java.io.IOException;
import java.util.concurrent.TimeUnit;

/**
 * This class allows you to launch an installer application, such as an updater, from your own application.
 * It includes a mechanism for the installer application to shut down the invoking application with a
 * "Shutdown calling launcher" action.
 *
 * <p>In order to use this class, please include {@code resource/i4jruntime.jar} from your
 * install4j installation into your class path. You do not need to distribute this file
 * along with your application, install4j will do this automatically for you.
 * </p>
 *
 * @author ej-technologies GmbH
 *
 */
public class ApplicationLauncher {

    /**
     * Launch an installer application that you have defined in the install4j IDE. The application is launched in a new JVM.
     * @param applicationId the ID of the installer application to be launched. You can show the IDs of all installer applications on the
     * "Installer-&gt;Screens &amp;actions" tab with the "Show IDs" tool bar button.
     * @param arguments the arguments you want to pass to the installer application. Can be {@code null} if no arguments
     * should be passed. Arguments like {@code -Dkey=value} will be passed as JVM parameters, just like when calling the installer
     * application from the command line.
     * @param blocking if this call should block until the installer application exits.
     * @param callback an optional call back for receiving notifications when the installer application exits or
     * when the installer application shuts down this JVM. Can be {@code null}
     * @throws IOException if there was an error starting the installer application
     */
    public static void launchApplication(String applicationId, String[] arguments, boolean blocking, Callback callback) throws IOException {
        ApplicationLauncherImpl.launchApplication(applicationId, arguments, blocking, callback);
    }

    /**
     * Launch an installer application that you have defined in the install4j IDE. The application is launched in the same process.
     * The look and feel as well as the locale is not changed. This call will return immediately if you call it from the event
     * dispatch thread (EDT), otherwise it will block until the installer application exits.
     * <p>
     * The "Shutdown calling launcher" action has a different effect than usual: The whole process will be terminated
     * when the installer application exits. Furthermore, the "Request privileges" action only works for elevating
     * a helper process, not the main process.
     * <p>
     * When using JavaFX in your application on macOS, make sure to pass the system property
     * {@code
     *   -Djavafx.embed.singleThread=true
     * }
     * to your launcher, otherwise the native event loops of JavaFX and the AWT may lead to a deadlock.
     *
     * @param applicationId the ID of the installer application to be launched. You can show the IDs of all installer applications on the
     * "Installer-&gt;Screens &amp;actions" tab with the "Show IDs" tool bar button.
     * @param arguments the arguments you want to pass to the installer application. Can be {@code null} if no arguments
     * should be passed. Arguments like {@code -Dkey=value} will be passed as JVM parameters, just like when calling the installer
     * application from the command line.
     * @param callback an optional call back for receiving notifications when the installer application exits or
     * when the installer application shuts down this JVM. Can be {@code null}
     * @param windowMode if the wizard should be shown in a dialog or in a frame
     * @param parentWindow the parent window if the wizard should be shown in a dialog. Can be {@code null}.
     * @throws IllegalStateException if another in-process installer application is currently running
     */
    public static void launchApplicationInProcess(String applicationId, String[] arguments, Callback callback, WindowMode windowMode, Window parentWindow) {
        ApplicationLauncherImpl.launchApplicationInProcess(applicationId, arguments, callback, windowMode, parentWindow);
    }

    /**
     * Causes the current thread to wait until the currently running in process
     * installer application has been finished. If no installer application is running,
     * the method returns immediately.
     *
     * @param timeout the maximum time to wait
     * @param unit the time unit of the {@code timeout} argument
     * @return {@code true} if the application has terminated, or no installer application was running and {@code false}
     *         if the waiting time elapsed before the count reached zero
     * @throws InterruptedException if the current thread is interrupted
     *         while waiting
     */
    public static boolean waitForInProcessApplication(long timeout, TimeUnit unit) throws InterruptedException {
        return ApplicationLauncherImpl.waitForInProcessApplication(timeout, unit);
    }

    /**
     * Determine if this is a new archive installation. An archive installation is defined by a unique build ID and an installation location.
     * If this method hasn't been called before for this archive installation it will return {@code true} during the lifetime of
     * the current process and set the "new" state to {@code false} for all future calls from other processes.
     *
     * <p>The state is stored as a user setting, so the installation will appear as "new" again if a launcher is executed with a different user account.
     * For initialization of global state, additional custom checks are necessary.</p>
     *
     * <p>When an installer was used this method will always return {@code false}.</p>
     *
     * @return {@code true} if this is a new archive installation
     */
    public static boolean isNewArchiveInstallation() {
        return FirstRun.isFirstRun(false);
    }

    /**
     * You can implement this interface to receive notifications when the installer application exits or needs to
     * shut down your application.
     * @see ApplicationLauncher#launchApplication(String, String[], boolean, ApplicationLauncher.Callback)
     */
    public interface Callback {
        /**
         * Invoked when the installer application has exited.
         * @param exitValue the exit value of the installer application
         */
        void exited(int exitValue);

        /**
         * Invoked before the installer application shuts down this JVM.
         * This happens when a "Shutdown calling launcher" action is executed in the
         * installer application.
         */
        void prepareShutdown();

        /**
         * Override this method and return an instance of {@link ProgressListener} to receive progress
         * information from the installer application. This is implemented on Windows, macOS and Linux.
         * On other Unix variants it is unsupported and can be switched on by setting the system property
         * {@code install4j.forceProgressListener=true}.
         */
        default ProgressListener createProgressListener() {
            return null;
        }
    }

    /**
     * You can implement this interface to receive progress information from the installer application.
     * An instance of this class has to be returned by {@link Callback#createProgressListener()}.
     */
    public interface ProgressListener {

        /**
         * A new screen has been activated.
         * @param id the ID of the screen
         */
        void screenActivated(String id);

        /**
         * A new action has started its execution.
         * @param id the ID of the action
         */
        void actionStarted(String id);

        /**
         * A status message was set in the progress interface.
         * @param message the new message.
         */
        void statusMessage(String message);

        /**
         * A detail message was set In the progress interface. This message is shown
         * below the status message.
         * @param message the new message.
         */
        void detailMessage(String message);

        /**
         * The percentage of completion of the current action has changed.
         * @param value a value between 0 and 100.
         */
        void percentCompleted(int value);

        /**
         * The percentage of completion of a secondary task, like the unpacking of a JAR file, has changed.
         * @param value a value between 0 and 100.
         */
        void secondaryPercentCompleted(int value);

        /**
         * The indeterminate progress state has changed.
         */
        void indeterminateProgress(boolean indeterminateProgress);
    }

    /**
     * Empty implementation of {@link ProgressListener}.
     * This can be used as a base class if only selected methods should be overridden.
     */
    public static class ProgressListenerAdapter implements ProgressListener {
        @Override
        public void screenActivated(String id) {

        }

        @Override
        public void actionStarted(String id) {

        }

        @Override
        public void statusMessage(String message) {

        }

        @Override
        public void detailMessage(String message) {

        }

        @Override
        public void percentCompleted(int value) {

        }

        @Override
        public void secondaryPercentCompleted(int value) {

        }

        @Override
        public void indeterminateProgress(boolean indeterminateProgress) {

        }
    }

    /**
     * Window mode for starting the application in process.
     * @see ApplicationLauncher#launchApplicationInProcess(String, String[], ApplicationLauncher.Callback, ApplicationLauncher.WindowMode, Window)
     */
    public enum WindowMode {
        /**
         * Shows the wizard in a modal dialog.
         */
        DIALOG,
        /**
         * Shows the wizard in a frame.
         */
        FRAME
    }
}
