package com.install4j.api.context;

import com.install4j.runtime.installer.InstallerConstants;

/**
 * Enumeration class that represents the different ways how install4j can react if asked to install a file over an already existing
 * file.
 * @author ej-technologies GmbH
 * @see FileOptions
 */
public enum OverwriteMode {

    /**
     * Indicates that if the destination file exists, it won't be replaced in any case.
     */
    NEVER("Never", InstallerConstants.OVERWRITE_NEVER),

    /**
     * Indicates that if the destination file exists, it will be replaced in any case.
     */
    ALWAYS("Always", InstallerConstants.OVERWRITE_ALWAYS),

    /**
     * Indicates that if the destination file exists, it will be replaced if the source
     * file is newer. On Windows, the file version resource will be used for the comparison.
     * If the destination file is newer, it will be skipped silently.
     */
    IF_NEWER("If newer", InstallerConstants.OVERWRITE_IF_NEWER),

    /**
     * Indicates that if the destination file exists, it will be replaced if the source
     * file is newer. On Windows, the file version resource will be used for the comparison.
     * If the destination file is newer, the framework will ask the user if the file should
     * be replaced.
     */
    IF_NEWER_OTHERWISE_ASK("If newer otherwise ask", InstallerConstants.OVERWRITE_IF_NEWER_OTHERWISE_ASK),

    /**
     * Indicates that if the destination file exists and has not been installed by a previous run of the installer,
     * the user will be asked whether to replace the file. Please note that
     * if the user chooses "Never" or "Always", the framework won't ask the user again.
     */
    ALWAYS_ASK_EXCEPT_FOR_UPDATE("Always ask except for update", InstallerConstants.OVERWRITE_ALWAYS_ASK_EXCEPT_FOR_UPDATE),

    /**
     * Indicates that the user will be asked whether to replace the file even if it was installed by a previous run
     * of the installer. Please note that if the user chooses "Never" or "Always", the framework won't ask the user again.
     */
    ALWAYS_ASK("Always ask", InstallerConstants.OVERWRITE_ALWAYS_ASK);

    /**
     * Convert int value to constant.
     * @param overwrite the int value
     * @return the constant
     */
    public static OverwriteMode getFromIntValue(int overwrite) {
        for (OverwriteMode overwriteMode : values()) {
            if (overwriteMode.intValue == overwrite) {
                return overwriteMode;
            }
        }
        return null;
    }

    private final String description;
    private final int intValue;

    OverwriteMode(String description, int value) {
        this.description = description;
        this.intValue = value;
    }

    /**
     * Convert to an int value.
     * @return the int value
     */
    public int getIntValue() {
        return intValue;
    }

    @Override
    public String toString() {
        return description;
    }

}
