package com.install4j.api.beaninfo;

import com.install4j.api.beans.ActionList;

import java.beans.IntrospectionException;

/**
 * Property descriptor for properties that contain a list of configurable actions.
 * The type of the property must be {@link ActionList}.
 * <p>Using this class is not strictly required. In principle, you could also set values for the {@code ATTRIBUTE_*} constants in the property descriptor
 * of an unrelated property descriptor class.
 * @see ActionList
 */
public class ActionListPropertyDescriptor extends Install4JPropertyDescriptor {

    /**
     * @see #setAllowedActionClasses(Class[])
     */
    public static final String ATTRIBUTE_ALLOWED_ACTION_CLASSES = "allowedActions";

    /**
     * @see #setExtraScriptParameters(ScriptParameter[])
     */
    public static final String ATTRIBUTE_EXTRA_SCRIPT_PARAMETERS = "extraScriptParameters";

    /**
     * @see #setConfigurableElevationType(boolean) (boolean)
     */
    public static final String ATTRIBUTE_CONFIGURABLE_ELEVATION_TYPE = "configurableElevationType";

    /**
     * @see #setAllowedActionClasses(Class[])
     */
    public static final String ATTRIBUTE_ACTION_KEYS = "actionsKeys";

    /**
     * Action key for properties of type {@link ActionList} that do not support rollback.
     * Properties in standard actions that deal with rollback are not shown in the install4j IDE in that case.
     * <p>Call {@link #setActionKeys(String[])} with an array that contains this string to select this behavior for
     * an action list property.</p>
     */
    public static final String ACTION_KEY_NO_ROLLBACK = "noRollback";

    /**
     * Create a property descriptor that can be passed to {@link Install4JBeanInfo#addPropertyDescriptor(Install4JPropertyDescriptor)}.
     * @param propertyName the name of the property
     * @param beanClass the class of the bean that contains the property. <b>Note:</b> This is <b>not</b> the class of the property.
     * @param displayName the display name of the property
     * @param shortDescription a short description of the property in HTML format. You do not have to start the description with &lt;html&gt;, it will be prepended automatically.
     * @return the property descriptor
     */
    public static ActionListPropertyDescriptor create(String propertyName, Class beanClass, String displayName, String shortDescription) {
        try {
            ActionListPropertyDescriptor descriptor = ActionListPropertyDescriptor.createDescriptor(propertyName, beanClass);
            descriptor.setDisplayName(displayName);
            descriptor.setShortDescription(shortDescription);
            return descriptor;
        } catch (IntrospectionException e) {
            throw new RuntimeException(e);
        }
    }

    private static ActionListPropertyDescriptor createDescriptor(String propertyName, Class beanClass) throws IntrospectionException {
        try {
            return new ActionListPropertyDescriptor(propertyName, beanClass);
        } catch (Exception ignored) {
        }
        return new ActionListPropertyDescriptor(propertyName, beanClass, "get" + capitalize(propertyName), null);
    }

    ActionListPropertyDescriptor(String propertyName, Class beanClass) throws IntrospectionException {
        super(propertyName, beanClass);
    }

    ActionListPropertyDescriptor(String propertyName, Class beanClass, String readMethod, String writeMethod) throws IntrospectionException {
        super(propertyName, beanClass, readMethod, writeMethod);
    }

    /**
     * If set to a non-null value, the action registry dialog will only display the specified actions.
     * Derived classes are also shown, so you can include entire hierarchies with a single class.
     * Otherwise, all actions will be offered.
     * @param actionClasses an array with the action classes
     * @return {@code this}, for chained calls on this property descriptor
     */
    public ActionListPropertyDescriptor setAllowedActionClasses(Class[] actionClasses) {
        if (actionClasses != null) {
            setValue(ATTRIBUTE_ALLOWED_ACTION_CLASSES, actionClasses);
        }
        return this;
    }

    /**
     * If set to a non-null value, script properties of any actions contained in this action list will receive
     * fixed extra script parameter of the defined type. The fixed parameters have to be passed to
     * {@link com.install4j.api.beans.AbstractBean#executeActionListAsync(ActionList, Object...)} or
     * {@link com.install4j.api.beans.AbstractBean#executeActionListSync(ActionList, Object...)} as the
     * vararg parameter.
     * @param scriptParameters the extra parameters
     * @return {@code this}, for chained calls on this property descriptor
     */
    public ActionListPropertyDescriptor setExtraScriptParameters(ScriptParameter[] scriptParameters) {
        if (scriptParameters != null) {
            setValue(ATTRIBUTE_EXTRA_SCRIPT_PARAMETERS, scriptParameters);
        }
        return this;
    }

    /**
     * By default, the user can configure the elevation type of the nested actions. In that case, nested actions
     * have an "Action elevation type" property that can be configured separately from the container.<p>
     * Sometimes it can be necessary to suppress this property, for example, if the actions depend on non-serializable
     * installer variables set by the container. To force a container action to be unelevated, you can call
     * {@code setDefaultActionElevationType(ActionElevationType.NONE, true)} in the bean config of the action.
     * </p>
     * @param configurableElevationType whether the action elevation type should be configurable
     * @return {@code this}, for chained calls on this property descriptor
     */
    public ActionListPropertyDescriptor setConfigurableElevationType(boolean configurableElevationType) {
        setValue(ATTRIBUTE_CONFIGURABLE_ELEVATION_TYPE, configurableElevationType);
        return this;
    }

    /**
     * Action keys can be used to show or hide properties of other actions.
     * See {@link Install4JPropertyDescriptor#setActionListShownKeys(String[])} and
     * {@link Install4JPropertyDescriptor#setActionListHiddenKeys(String[])} for more information.
     * @param actionsKeys an array with the action keys
     * @return {@code this}, for chained calls on this property descriptor
     */
    public ActionListPropertyDescriptor setActionKeys(String[] actionsKeys) {
        if (actionsKeys != null) {
            setValue(ATTRIBUTE_ACTION_KEYS, actionsKeys);
        }
        return this;
    }
}
