package com.install4j.api;

import com.install4j.api.context.Context;
import com.install4j.api.context.UserCanceledException;
import com.install4j.runtime.installer.ContextImpl;
import com.install4j.runtime.installer.ContextInt;
import com.install4j.runtime.installer.InstallerVariables;
import com.install4j.runtime.installer.MediaPlatformType;
import com.install4j.runtime.installer.frontend.GUIHelper;
import com.install4j.runtime.installer.helper.Install4jClassLoader;
import com.install4j.runtime.installer.helper.InstallerUtil;
import com.install4j.runtime.installer.helper.Logger;
import com.install4j.runtime.installer.helper.LoggerImpl;
import com.install4j.runtime.installer.helper.comm.ExecutionContext;
import com.install4j.runtime.installer.helper.comm.HelperCommunication;
import com.install4j.runtime.installer.helper.content.ContentInstaller;
import com.install4j.runtime.installer.platform.UserInfo;
import org.jetbrains.annotations.Nls;

import javax.swing.*;
import java.awt.*;
import java.io.File;
import java.io.IOException;
import java.net.URL;

/**
 * The class contains various utility functions to be used by
 * script properties, custom actions and custom screens.
 *
 * @author ej-technologies GmbH
 */
public class Util {

    /**
     * Returns the home directory.
     * @return the home directory
     */
    public static String getUserHome() {
        String userHome = InstallerVariables.getStringVariable(InstallerVariables.VARIABLE_USER_HOME);
        if (userHome == null) {
            return System.getProperty("user.home");
        } else {
            return userHome;
        }
    }

    /**
     * Returns whether the platform is Solaris.
     *
     * @return the result
     */
    public static boolean isSolaris() {
        return InstallerUtil.isSolaris();
    }

    /**
     * Returns whether the platform is Linux.
     *
     * @return the result
     */
    public static boolean isLinux() {
        return InstallerUtil.isLinux();
    }

    /**
     * Returns whether the platform is Windows.
     *
     * @return the result
     */
    public static boolean isWindows() {
        return InstallerUtil.isWindows();
    }

    /**
     * Returns whether the platform is Windows 9X (95, 98, ME).
     *
     * @return the result
     * @deprecated always returns {@code false} because launchers do not work with Windows 9X anymore
     */
    @Deprecated
    public static boolean isWindows9X() {
        return false;
    }

    /**
     * Returns whether the platform is Windows NT.
     *
     * @return the result
     */
    @Deprecated
    public static boolean isWindowsNT() {
        return InstallerUtil.isWindowsNT();
    }

    /**
     * Returns whether the platform is at least Windows Vista.
     *
     * @return the result
     */
    public static boolean isAtLeastWindowsVista() {
        return InstallerUtil.isAtLeastWindowsVista();
    }

    /**
     * Returns whether the platform is at least Windows 7.
     *
     * @return the result
     */
    public static boolean isAtLeastWindows7() {
        return InstallerUtil.isAtLeastWindows7();
    }

    /**
     * Returns whether the platform is at least Windows XP.
     *
     * @return the result
     */
    @Deprecated
    public static boolean isAtLeastWindowsXP() {
        return isWindows();
    }

    /**
     * Returns whether the platform is Windows Vista.
     *
     * @return the result
     */
    public static boolean isWindowsVista() {
        return InstallerUtil.isWindowsVista();
    }

    /**
     * Returns whether the platform is Windows 7.
     *
     * @return the result
     */
    public static boolean isWindows7() {
        return InstallerUtil.isWindows7();
    }

    /**
     * Returns whether the platform is Windows 8. This includes Windows 8.1.
     *
     * @return the result
     */
    public static boolean isWindows8() {
        return InstallerUtil.isWindows8();
    }

    /**
     * Returns whether the platform is Windows 10.
     *
     * @return the result
     */
    public static boolean isWindows10() {
        return InstallerUtil.isWindows10();
    }

    /**
     * Returns whether the platform is at least Windows 10.
     *
     * @return the result
     */
    public static boolean isAtLeastWindows10() {
        return InstallerUtil.isAtLeastWindows10();
    }

     /**
     * Returns whether the platform is Windows XP.
     *
     * @return the result
     */
    public static boolean isWindowsXP() {
        return InstallerUtil.isWindowsXP();
    }

    /**
     * Returns whether the platform is Windows 2000.
     *
     * @return the result
     */
    public static boolean isWindows2000() {
        return InstallerUtil.isWindows2000();
    }

    /**
     * Returns whether the platform is Windows 2003.
     *
     * @return the result
     */
    public static boolean isWindows2003() {
        return InstallerUtil.isWindows2003();
    }

    /**
     * Returns whether the platform is Windows 2008.
     *
     * @return the result
     */
    public static boolean isWindows2008() {
        return InstallerUtil.isWindows2008();
    }

    /**
     * Returns whether the platform is Windows 2012.
     *
     * @return the result
     */
    public static boolean isWindows2012() {
        return InstallerUtil.isWindows2012();
    }

    /**
     * Returns whether the platform is Windows 2016.
     *
     * @return the result
     */
    public static boolean isWindows2016() {
        return InstallerUtil.isWindows2016();
    }

    /**
     * Returns whether the Windows is a 64-bit Windows, regardless of whether the installer is running with a 32-bit JVM or
     * a 64-bit JVM. This condition cannot be found out by inspecting {@code System.getProperty("os.arch")}, since this will
     * return a value that corresponds to the JRE and not to the OS, for example, always "x86" for a 32-bit JRE, even if it is running
     * on a 64-bit Windows.
     * <p>
     * This method inspects the environment variables {@code PROCESSOR_ARCHITECTURE} and {@code PROCESSOR_ARCHITEW6432}.
     * If {@code PROCESSOR_ARCHITECTURE} contains {@code AMD64}, {@code IA64} or {@code ARM64}, the currently used JRE is a 64-bit JRE which is then
     * by definition running on a 64-bit Windows. If {@code PROCESSOR_ARCHITECTURE} contains {@code x86}, the {@code PROCESSOR_ARCHITEW6432}
     * is undefined for a 32-bit Windows and contains {@code AMD64}, {@code IA64} or {@code ARM64} for a 64-bit Windows.
     * @return the result. If the current OS is not Windows, the result is always {@code false}.
     */
    public static boolean is64BitWindows() {
        return isWindows() && (!InstallerUtil.is32BitJVM() || InstallerUtil.is64BitWindows());
    }

    /**
     * Returns whether the platform is macOS.
     *
     * @return the result
     */
    public static boolean isMacOS() {
        return InstallerUtil.isMacOS();
    }

    /**
     * Returns whether the platform is HP UX.
     *
     * @return the result
     */
    public static boolean isHpux() {
        return InstallerUtil.isHpux();
    }

    /**
     * Returns whether the platform is AIX.
     *
     * @return the result
     */
    public static boolean isAix() {
        return InstallerUtil.isAix();
    }

    /**
     * Returns whether the installer is a Windows installer
     *
     * @return the result
     */
    public static boolean isWindowsInstaller() {
        return InstallerUtil.getMediaPlatformType() == MediaPlatformType.WINDOWS;
    }

    /**
     * Returns whether the installer is a Unix installer
     *
     * @return the result
     */
    public static boolean isUnixInstaller() {
        return InstallerUtil.getMediaPlatformType() == MediaPlatformType.UNIX;
    }

    /**
     * Returns whether the installer is a macOS installer
     *
     * @return the result
     */
    public static boolean isMacosInstaller() {
        return InstallerUtil.getMediaPlatformType() == MediaPlatformType.MACOS;
    }

    /**
     * Returns the standard directory for installing applications.
     *
     * @return the directory
     */
    public static String getStandardApplicationsDirectory() {
        return InstallerUtil.getStandardApplicationsDirectory(false);
    }

    /**
     * Returns whether the application was installed from an archive or an installer media type.
     *
     * @return {@code true} if this is an archive
     */
    public static boolean isArchive() {
        return InstallerUtil.isArchive();
    }

    /**
     * Show a URL in the default browser. If no default browser can be
     * determined for the operating system, the user is asked to locate the
     * executable of the internet browser. This executable is cached so that
     * subsequent invocations of this method do not bring up this question again.
     * <p>
     * For quiet installers, this method does nothing.
     * </p>
     *
     * @param url the URL to be shown in the browser.
     */
    public static void showUrl(final URL url) {
        HelperCommunication.getInstance().runAction(ExecutionContext.UNELEVATED, context -> GUIHelper.showURL(url.toExternalForm()));
    }

    /**
     * Show a message dialog. This message dialog is a JOptionPane, the parameters are
     * equivalent to those of the standard Java class. This method works in both GUI and console mode.
     *
     * @param message     the message
     * @param messageType the message type (see JOptionPane)
     * @see JOptionPane
     */
    public static void showMessage(@Nls final String message, final int messageType) {
        HelperCommunication.getInstance().runAction(ExecutionContext.UNELEVATED, context -> GUIHelper.showMessage(null, message, messageType));
    }

    /**
     * Show a message dialog. This message dialog is a JOptionPane with a message type of INFORMATION_MESSAGE.
     * This method works in both GUI and console mode.
     * <p>
     * Paragraphs can be created by double line feeds (\n\n), single line feeds will be ignored.
     * On Windows and macOS, the string is split into a title that is shown in a larger font and a message that
     * is shown in a regular font. If the message does not contain a line feed, the entire message is shown as a title.
     * If the message contains a line feed, the title is split from the beginning up to the first dot or the first line
     * feed. The remainder is shown as the regular message. If you do not want to display anything in the title font,
     * start the message with a single line break.
     * </p>
     * <p>
     * Text is wrapped automatically on word boundaries. On Windows, text that cannot be wrapped is split with an
     * ellipsis in the middle of the line to limit the maximum width of the message dialog. In some circumstances this
     * may not be desirable, for example, for displaying stack trace elements. You can prepend the message with a
     * zero character {@code \0} to prevent word wrapping and ellipsis insertion. On macOS, text is wrapped on word
     * boundaries and within words if the width of a single word is too large. This behavior of the native alert
     * cannot be changed.
     * </p>
     *
     * @param message the message
     * @see JOptionPane
     */
    public static void showMessage(@Nls String message) {
        showMessage(message, JOptionPane.INFORMATION_MESSAGE);
    }

    /**
     * Show a message dialog. This message dialog is a JOptionPane with a message type of ERROR_MESSAGE.
     * This method works in both GUI and console mode.
     * <p>
     * See {@link #showMessage(String, int)} for a discussion on the message format.
     * </p>
     *
     * @param message the message
     * @see JOptionPane
     */
    public static void showErrorMessage(@Nls String message) {
        showMessage(message, JOptionPane.ERROR_MESSAGE);
    }

    /**
     * Show a message dialog. This message dialog is a JOptionPane with a message type of WARNING_MESSAGE.
     * This method works in both GUI and console mode.
     * <p>
     * See {@link #showMessage(String, int)} for a discussion on the message format.
     * </p>
     *
     * @param message the message
     * @see JOptionPane
     */
    public static void showWarningMessage(@Nls String message) {
        showMessage(message, JOptionPane.WARNING_MESSAGE);
    }

    /**
     * Show an option dialog. This option dialog is a JOptionPane, the parameters are
     * equivalent to those of the standard Java class. This method works in both GUI and console mode.
     * <p>
     * See {@link #showMessage(String, int)} for a discussion on the message format.
     * </p>
     *
     * @param message     the message
     * @param options     the options. To explicitly set keys for answers in console mode, prefix the desired letter
     *                    with '&amp;'. For example {@code new String[] {"&amp;Enter", "E&amp;xit"}}. If no explicit keys are set,
     *                    the first letters of the options are taken. Repeating letters are replaced with numeric indices.
     * @param messageType the message type (see JOptionPane)
     * @return the index of the selected option or -1 if the installer is running in unattended mode
     * @throws UserCanceledException if the user cancels the selection in console mode
     * @see JOptionPane
     */
    public static int showOptionDialog(@Nls final String message, @Nls final String[] options, final int messageType) throws UserCanceledException {
        try {
            return HelperCommunication.getInstance().fetchIntChecked(ExecutionContext.UNELEVATED,
                    context -> GUIHelper.showOptionDialog(null, message, null, options, messageType));
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Alias for @{link Context#handleCriticalException(Throwable)}.
     *
     */
    public static void fatalError(final Throwable t) {
        try {
            HelperCommunication.getInstance().runAction(ExecutionContext.UNELEVATED, context -> {
                System.setProperty(Logger.PROPNAME_KEEP_LOGFILE, "true");
                InstallerUtil.reportException(t);
                ContextInt contextInt = ContextImpl.getSingleContextInt();
                if (contextInt != null) {
                    contextInt.immediateExit(1);
                } else {
                    InstallerUtil.exit(1);
                }
            });
        } catch (Throwable throwable) {
            throwable.printStackTrace();
        }
    }

    /**
     * Get the window of the installer application.
     * @deprecated use {@link UiUtil#getParentWindow()} instead
     */
    @Deprecated
    public static Window getParentWindow() {
        return UiUtil.getParentWindow();
    }

    /**
     * Writes an info message to the installation log.
     *
     * @param source the object that requests the log. Usually the bean that executes this code. Can be {@code null}, in this
     * case the currently executed action or - if no action is being executed - the currently displayed screen is used. A
     * case where you should explicitly set this parameter is in code that is executed from form components.
     * @param message the message to be logged.
     */
    public static void logInfo(Object source, String message) {
        Logger.getInstance().info(source, message);
    }

    /**
     * Writes an error message to the installation log.
     *
     * @param source the object that requests the log. Usually the bean that executes this code. Can be {@code null}, in this
     * case the currently executed action or - if no action is being executed - the currently displayed screen is used. A
     * case where you should explicitly set this parameter is in code that is executed from form components.
     * @param message the message to be logged.
     */
    public static void logError(Object source, String message) {
        Logger.getInstance().error(source, message);
    }

    /**
     * Writes a stack trace to the installation log.
     *
     * @param t the throwable instance
     */
    public static void log(Throwable t) {
        Logger.getInstance().log(t);
    }

    /**
     * Dump all defined installer variables to stderr.
     * To be able to see the output, you have to create a debug installer by selecting
     * the corresponding option in the Build step in the install4j GUI.
     *
     * @param context the context
     */
    public static void dumpVariables(Context context) {
        System.err.println();
        System.err.println("Defined installer variables:");
        System.err.println("----------------------------");
        for (String variableName : context.getVariableNames()) {
            Object value = context.getVariable(variableName);
            System.err.println(InstallerUtil.getVerboseVariableDefinition(variableName, value));
        }
        System.err.println("----------------------------");
        System.err.println();
    }

    /**
     * Checks whether the current user has administration privileges.
     * <ul>
     * <li>On Windows this method returns {@code true} if the installer is running with administration privileges.
     * <li>On macOS this method returns {@code true} if the user is member of the "admin" group.</li>
     * <li>On Unix this method returns {@code true} if the current user is called "root".</li>
     * </ul>
     *
     * @return {@code true} or {@code false}.
     * @deprecated use {@link #isAdminGroup} and {@link #hasFullAdminRights} instead.
     */
    @Deprecated
    public static boolean isAdminUser() {
        return UserInfo.isAdminUser();
    }

    /**
     * Checks whether the current user is from the administration group.
     * <ul>
     * <li>On Windows this is the case if the user is a member of the "Administrators" group.</li>
     * <li>On macOS this method returns {@code true} if the user is member of the "admin" group.</li>
     * <li>On Unix this method returns {@code true} if the current user is called "root".</li>
     * </ul>
     *
     * @return {@code true} or {@code false}.
     */
    public static boolean isAdminGroup() {
        return UserInfo.isAdminGroup();
    }

    /**
     * Checks whether the current process has full administration rights.
     * <ul>
     * <li>On Windows this method returns {@code true} if the installer is running with elevated privileges.</li>
     * <li>On macOS and Unix this method returns {@code true} if the current user is called "root".</li>
     * </ul>
     *
     * @return {@code true} or {@code false}.
     */
    public static boolean hasFullAdminRights() {
        return UserInfo.hasFullAdminRights();
    }

    /**
     * Checks whether the current process has the necessary rights to perform an installation for all users.
     * <p>
     * This is the case if the installer has been started with elevated privileges or if a "Request privileges" action
     * started the elevated helper process. On macOS, it is already sufficient if the user is in the admin group.
     * </p>
     * @return {@code true} or {@code false}.
     */
    public static boolean hasAllUserInstallationRights() {
        return (isMacOS() ? isAdminGroup() : hasFullAdminRights()) || HelperCommunication.getInstance().hasElevatedHelper() || HelperCommunication.getInstance().isElevatedHelper();
    }

    /**
     * If a stack trace comes from scripts, it can be difficult to trace them to their origin by looking at the
     * stack trace alone. With this method, you can annotate the stack trace with the actual names of properties,
     * actions, screens or form components that contain the offending script code.
     *
     * @param t the exception
     * @return the annotated stack trace
     */
    public static String getAnnotatedStackTrace(Throwable t) {
        return InstallerUtil.getAnnotatedStackTrace(t);
    }

    /**
     * Prints the return value of {@link #getAnnotatedStackTrace(Throwable)} to {@code System.err}.
     * This can be used for debugging purposes.
     *
     * @param t the exception
     */
    public static void printAnnotatedStackTrace(Throwable t) {
        System.err.print(getAnnotatedStackTrace(t));
    }

    /**
     * Show the specified directory or file. On Windows this will open an explorer window, and on Linux/Unix it will open a terminal.
     * On macOS it will open a finder window for regular files or directories.
     * @param path the directory that should be displayed
     */
    public static void showPath(final String path) {
        HelperCommunication.getInstance().runAction(ExecutionContext.UNELEVATED, context -> GUIHelper.showPath(path));
    }

    /**
     * Checks if a directory is writable with the currently available privileges.
     * @param directory the directory. If the directory is a file, {@code false} is returned. It is acceptable if the
     *                  directory does not exist.
     * @return {@code true} or {@code false}.
     * */
    public static boolean isDirectoryWritable(final File directory) {
        return HelperCommunication.getInstance().fetchBoolean(ContentInstaller.getExecutionContext(), context -> InstallerUtil.checkWritable(directory));
    }

    /**
     * Get the location of the log file.
     * This is a temporary file, for the installer it will be moved to {@code .install4j/installation.log}
     * before the installer terminates if an "Install files" action has been executed. For an installer application,
     * this log file will be deleted when the installer application is terminated unless the VM parameter
     * {@code -Dinstall4j.keepLog=true} is set.
     *
     * @return the log file
     */
    public static File getLogFile() {
        return HelperCommunication.getInstance().fetchObject(ExecutionContext.UNELEVATED, context -> ((LoggerImpl)LoggerImpl.getInstance()).getLogFile());
    }

    /**
     * Move the log file to a new location.
     * With this method, an alternative permanent location of the log file can be specified. The log file
     * will be moved immediately, and it will not be deleted when the installer application exits.
     * @see #getLogFile
     */
    public static void moveLogFile(final File newLogFile) {
        HelperCommunication.getInstance().runAction(ExecutionContext.UNELEVATED, context -> ((LoggerImpl)LoggerImpl.getInstance()).moveLogFile(newLogFile, true));
    }

    /**
     * Call {@link System#load} with the supplied file name resolved against the resource directory.
     * This is useful to load supporting native libraries for SQL drivers, such as
     * {@code sqljdbc_auth.dll} for the Oracle JDBC driver. If you add that DLL on the "Installer-&gt;Custom code &amp;
     * resources step", you can load it with the code
     * <pre>{@code
     *     Util.loadNativeFromResources("sqljdbc_auth.dll");
     * }</pre>
     * in a "Run script" action.
     * @param fileName the file name of the native library
     */
    public static void loadNativeFromResources(String fileName) {
        String absoluteFileName = InstallerVariables.getStringVariable(InstallerVariables.VARIABLE_RESOURCE_DIR) + File.separator + fileName;
        try {
            Install4jClassLoader.invokeInCustomClassLoader(System.class.getMethod("load", String.class), null, absoluteFileName);
        } catch (Throwable t) {
            throw new RuntimeException(t);
        }
    }
}
