package com.install4j.api.screens;

import com.install4j.api.context.ProgressInterface;
import com.install4j.api.context.UserCanceledException;
import org.jetbrains.annotations.Nls;

/**
 * A {@code Console} object allows you to display information to the user and request input when the installer or uninstaller is running in console mode.
 * Installers or uninstallers must be started with the "-c" argument to run in console mode.
 * A console object is passed to the {@code handleConsole} methods of screens and form components.
 *
 * @see Screen#handleConsole(Console)
 * @see com.install4j.api.formcomponents.FormComponent#handleConsole(Console)  
 * @author ej-technologies GmbH
 */
public interface Console extends ProgressInterface {

    /**
     * Print an object to the console by invoking its {@code toString()} method.
     * @param val the object to be printed
     */
    void print(Object val);
    /**
     * Print an object to the console by invoking its {@code toString()} method and append a line feed.
     * @param val the object to be printed
     */
    void println(Object val);

    /**
     * Print a line feed to the console.
     */
    void println();

    /**
     * Print a long text to the console and partition it into pages.
     * Line breaks are introduced after 75 characters, and a "more" notice is printed
     * every 25 lines.
     * @param text the text
     * @throws UserCanceledException if the user cancels the "more" notice, typically with CTRL-C
     */
    void more(@Nls String text) throws UserCanceledException;

    /**
     * Read input from the console.
     * @return the string that was entered by the user.
     * @throws UserCanceledException if the user cancels the prompt, typically with CTRL-C
     */
    String readLine() throws UserCanceledException;

    /**
     * Ask the user a question that has to be answered with "Yes" or "No".
     * There is no default value.
     * @param message the question
     * @return whether the user answered "Yes"
     * @throws UserCanceledException if the user cancels the question, typically with CTRL-C
     */
    boolean askYesNo(@Nls String message) throws UserCanceledException;

    /**
     * Ask the user a question that has to be answered with "Yes" or "No".
     * The default value is configurable.
     * @param message the question
     * @param defaultValue whether "Yes" is the default value
     * @return whether the user answered "Yes"
     * @throws UserCanceledException if the user cancels the question, typically with CTRL-C
     */
    boolean askYesNo(@Nls String message, boolean defaultValue) throws UserCanceledException;

    /**
     * Ask the user a question that has to be answered with "OK" or "Cancel".
     * There is no default value.
     * @param message the question
     * @return whether the user answered "OK"
     * @throws UserCanceledException if the user cancels the question, typically with CTRL-C.
     *  Note that this is different from a "Cancel" answer.
     */
    boolean askOkCancel(@Nls String message) throws UserCanceledException;

    /**
     * Ask the user a question that has to be answered with "OK" or "Cancel".
     * The default value is configurable.
     * @param message the question
     * @param defaultValue whether "OK" is the default value
     * @return whether the user answered "OK"
     * @throws UserCanceledException if the user cancels the question, typically with CTRL-C.
     *  Note that this is different from a "Cancel" answer.
     */
    boolean askOkCancel(@Nls String message, boolean defaultValue) throws UserCanceledException;

    /**
     * Ask the user a question with a fixed set of answers.
     * There is no default value.
     * @param message the question
     * @param descriptions an array with a textual description for each option
     * @param keys an array with a short key for each option. Keys are what the user enters to select the desired option.
     * @return the selected index of the option
     * @throws UserCanceledException if the user cancels the question, typically with CTRL-C.
     */
    int askOption(@Nls String message, @Nls String[] descriptions, String[] keys) throws UserCanceledException;

    /**
     * Ask the user a question with a fixed set of answers.
     * The default value is configurable.
     * @param message the question
     * @param descriptions an array with a textual description for each option
     * @param keys an array with a short key for each option. Keys are what the user enters to select the desired option.
     * @param defaultValue the index of the default option
     * @return the selected index of the option
     * @throws UserCanceledException if the user cancels the question, typically with CTRL-C.
     */
    int askOption(@Nls String message, @Nls String[] descriptions, String[] keys, int defaultValue) throws UserCanceledException;

    /**
     * Ask the user a question with a fixed set of answers.
     * The default value, the no-selection policy as well as the presentation of the options are configurable.
     * @param message the question
     * @param descriptions an array with a textual description for each option
     * @param keys an array with a short key for each option. Keys are what the user enters to select the desired option.
     * @param defaultValue the index of the default option
     * @param allowNoSelection if the question should allow an empty answer, the return value is -1 in that case.
     * @param oneLinePerOption if the options should be presented on option per line. By default, all options are presented on the same line.
     * @return the index of the selected option or -1 if no answer is given.
     * @throws UserCanceledException if the user cancels the question, typically with CTRL-C.
     */
    int askOption(@Nls String message, @Nls String[] descriptions, String[] keys, int defaultValue, boolean allowNoSelection, boolean oneLinePerOption) throws UserCanceledException;

    /**
     * Ask the user a question with a fixed set of answers.
     * The default value, the help texts, the no-selection policy as well as the presentation of the options are configurable.
     * @param message the question
     * @param descriptions an array with a textual description for each option
     * @param keys an array with a short key for each option. Keys are what the user enters to select the desired option.
     * @param helpTexts an array with help texts that can be shown by the user, array elements can be null for no help
     * @param defaultValue the index of the default option
     * @param allowNoSelection if the question should allow an empty answer, the return value is -1 in that case.
     * @param oneLinePerOption if the options should be presented on option per line. By default, all options are presented on the same line.
     * @return the index of the selected option or -1 if no answer is given.
     * @throws UserCanceledException if the user cancels the question, typically with CTRL-C.
     */
    int askOption(@Nls String message, @Nls String[] descriptions, String[] keys, @Nls String[] helpTexts, int defaultValue, boolean allowNoSelection, boolean oneLinePerOption) throws UserCanceledException;

    /**
     * Ask the user to select multiple elements from a list of options.
     * The default selection and the presentation of the options are configurable.
     * @param message the question
     * @param descriptions an array with a textual description for each option
     * @param keys an array with a short key for each option. Keys are what the user enters to select the desired options.
     * @param defaultValues an array with the indices of the options that are selected by default
     * @param oneLinePerOption if the options should be presented on option per line. By default, all options are presented on the same line.
     * @return an array with the indices of the selected options. The length of the array can be zero if no options were selected.
     * @throws UserCanceledException if the user cancels the question, typically with CTRL-C.
     */
    int[] askMultipleSelection(@Nls String message, @Nls String[] descriptions, String[] keys, int[] defaultValues, boolean oneLinePerOption) throws UserCanceledException;

    /**
     * Ask the user to enter a string with a default string that can be accepted by the user.
     * @param message the prompt before the input
     * @param defaultValue the default string
     * @return the string entered by the user or the default value if the user hits the Enter key.
     * @throws UserCanceledException if the user cancels the question, typically with CTRL-C.
     */
    String askString(@Nls String message, String defaultValue) throws UserCanceledException;

    /**
     * Ask the user to enter a password.
     * @param message the prompt before the input
     * @param defaultValue the default string
     * @return the string entered by the user or the default value if the user hits the Enter key.
     * @throws UserCanceledException if the user cancels the question, typically with CTRL-C.
     */
    String askPassword(@Nls String message, String defaultValue) throws UserCanceledException;

    /**
     * Print [Enter] to the console and wait for the user to hit the Enter key.
     * @throws UserCanceledException if the user cancels the notice, typically with CTRL-C.
     */
    void waitForEnter() throws UserCanceledException;

    /**
     * Print the header for a new screen. This invokes the "Console screen change handler"
     * that is set for the application node on the Screens &amp; Actions tab.
     * If that script is not configured, nothing will be printed.
     * @param title the title
     * @throws UserCanceledException if the user cancels the notice, typically with CTRL-C.
     */
    void printHeader(@Nls String title) throws UserCanceledException;
}
