package com.install4j.api.formcomponents;

import javax.swing.*;
import javax.swing.text.JTextComponent;

/**
 * With the widget style handler, you can apply a widget style to certain Swing components.
 * This is only required if you create buttons in a custom form component or custom screen.
 * <p>
 *  All methods take a widget style ID. You can pass the empty string for the widget style that has been selected for the current screen style.
 *  If you want the user to select a widget style, you can add a property of type {@code String} to your bean with a context of
 *  {@link com.install4j.api.beaninfo.Install4JPropertyDescriptor#CONTEXT_WIDGET_STYLE_ID CONTEXT_WIDGET_STYLE_ID}.
 *  The user can also select the inherited style, in which case the property will contain the empty string.
 * </p>
 * @see FormEnvironment#getWidgetStyleHandler()
 */
public interface WidgetStyleHandler {
    /**
     * Apply a widget style to a button.
     *
     * @param button the button
     * @param widgetStyleId the widget style ID. Pass the empty string for the default widget style of the current screen style.
     * @return the button, for fluent invocations
     */
    <T extends JButton> T applyWidgetStyle(T button, String widgetStyleId);

    /**
     * Apply a widget style to a text component.
     *
     * @param textComponent the text component
     * @param widgetStyleId the widget style ID. Pass the empty string for the default widget style of the current screen style.
     * @return the text component, for fluent invocations
     */
    <T extends JTextComponent> T applyWidgetStyle(T textComponent, String widgetStyleId);

    /**
     * Apply a widget style to a spinner.
     *
     * @param spinner the spinner
     * @param widgetStyleId the widget style ID. Pass the empty string for the default widget style of the current screen style.
     * @return the spinner, for fluent invocations
     */
    <T extends JSpinner> T applyWidgetStyle(T spinner, String widgetStyleId);

    /**
     * Apply a widget style to a combo box.
     *
     * @param comboBox the combo box
     * @param widgetStyleId the widget style ID. Pass the empty string for the default widget style of the current screen style.
     * @return the combo box, for fluent invocations
     */
    <T extends JComboBox> T applyWidgetStyle(T comboBox, String widgetStyleId);

    /**
     * Apply a widget style to a label.
     *
     * @param label the label
     * @param widgetStyleId the widget style ID. Pass the empty string for the default widget style of the current screen style.
     * @return the label, for fluent invocations
     */
    <T extends JLabel> T applyWidgetStyle(T label, String widgetStyleId);

    /**
     * Apply a widget style to a toggle button.
     *
     * @param toggleButton  the toggle button
     * @param widgetStyleId the widget style ID. Pass the empty string for the default widget style of the current screen style.
     * @return the toggle button, for fluent invocations
     */
    <T extends JToggleButton> T applyWidgetStyle(T toggleButton, String widgetStyleId);

    /**
     * Apply a widget style to a list.
     *
     * @param list the list
     * @param widgetStyleId the widget style ID. Pass the empty string for the default widget style of the current screen style.
     * @return the list, for fluent invocations
     */
    <T extends JList> T applyWidgetStyle(T list, String widgetStyleId);

    /**
     * Apply a widget style to a tree.
     *
     * @param tree the tree
     * @param widgetStyleId the widget style ID. Pass the empty string for the default widget style of the current screen style.
     * @return the tree, for fluent invocations
     */
    <T extends JTree> T applyWidgetStyle(T tree, String widgetStyleId);

    /**
     * Apply a widget style to a slider.
     *
     * @param slider the slider
     * @param widgetStyleId the widget style ID. Pass the empty string for the default widget style of the current screen style.
     * @return the slider, for fluent invocations
     */
    <T extends JSlider> T applyWidgetStyle(T slider, String widgetStyleId);

    /**
     * Apply a widget style to a progress bar.
     *
     * @param progressBar the progress bar
     * @param widgetStyleId the widget style ID. Pass the empty string for the default widget style of the current screen style.
     * @return the progress bar, for fluent invocations
     */
    <T extends JProgressBar> T applyWidgetStyle(T progressBar, String widgetStyleId);

    /**
     * Apply the default widget style to a button.
     * @see #applyWidgetStyle(JButton, String)
     * @param button the button
     * @return the button, for fluent invocations
     */
    default <T extends JButton> T applyWidgetStyle(T button) {
        return applyWidgetStyle(button, "");
    }

    /**
     * Apply the default widget style to a text component.
     * @see #applyWidgetStyle(JTextComponent, String)
     * @param textComponent the text component
     * @return the text component, for fluent invocations
     */
    default <T extends JTextComponent> T applyWidgetStyle(T textComponent) {
        return applyWidgetStyle(textComponent, "");
    }

    /**
     * Apply the default widget style to a spinner.
     * @see #applyWidgetStyle(JSpinner, String)
     * @param spinner the spinner
     * @return the spinner, for fluent invocations
     */
    default <T extends JSpinner> T applyWidgetStyle(T spinner) {
        return applyWidgetStyle(spinner, "");
    }

    /**
     * Apply the default widget style to a combo box.
     * @see #applyWidgetStyle(JComboBox, String)
     * @param comboBox the combo box
     * @return the combo box, for fluent invocations
     */
    default <T extends JComboBox> T applyWidgetStyle(T comboBox) {
        return applyWidgetStyle(comboBox, "");
    }

    /**
     * Apply the default widget style to a label.
     * @see #applyWidgetStyle(JLabel, String)
     * @param label the label
     * @return the label, for fluent invocations
     */
    default <T extends JLabel> T applyWidgetStyle(T label) {
        return applyWidgetStyle(label, "");
    }

    /**
     * Apply the default widget style to a toggle button.
     * @see #applyWidgetStyle(JToggleButton, String)
     * @param toggleButton the toggle button
     * @return the toggle button, for fluent invocations
     */
    default <T extends JToggleButton> T applyWidgetStyle(T toggleButton) {
        return applyWidgetStyle(toggleButton, "");
    }

    /**
     * Apply the default widget style to a list.
     * @see #applyWidgetStyle(JList, String)
     * @param list the list
     * @return the list, for fluent invocations
     */
    default <T extends JList> T applyWidgetStyle(T list) {
        return applyWidgetStyle(list, "");
    }

    /**
     * Apply the default widget style to a tree.
     * @see #applyWidgetStyle(JTree, String)
     * @param tree the tree
     * @return the tree, for fluent invocations
     */
    default <T extends JTree> T applyWidgetStyle(T tree) {
        return applyWidgetStyle(tree, "");
    }

    /**
     * Apply the default widget style to a slider.
     * @see #applyWidgetStyle(JSlider, String)
     * @param slider the slider
     * @return the slider, for fluent invocations
     */
    default <T extends JSlider> T applyWidgetStyle(T slider) {
        return applyWidgetStyle(slider, "");
    }

    /**
     * Apply the default widget style to a progress bar.
     * @see #applyWidgetStyle(JProgressBar, String)
     * @param progressBar the progress bar
     * @return the progress bar, for fluent invocations
     */
    default <T extends JProgressBar> T applyWidgetStyle(T progressBar) {
        return applyWidgetStyle(progressBar, "");
    }
}
