package com.install4j.api.beaninfo;

import com.install4j.api.actions.Action;

import java.beans.BeanDescriptor;

/**
 * Base class for BeanInfos of actions classes.
 * <p>Using this class is not strictly required. In principle, you could also set values for the {@code ATTRIBUTE_*} constants in the bean descriptor
 * of an unrelated bean info class.
 */
public abstract class ActionBeanInfo extends Install4JBeanInfo {

    /**
     * @see #setAssociatedConfirmation(String, String, String)
     */
    public static final String ATTRIBUTE_ASSOCIATED_CONFIRMATION = "associatedConfirmation";

    /**
     * @see #setAssociatedConfirmation(String, String, String)
     */
    public static final String ATTRIBUTE_ASSOCIATED_CONFIRMATION_INIT_SCRIPT = "associatedConfirmationInitScript";

    /**
     * @see #setAssociatedConfirmation(String, String, String)
     */
    public static final String ATTRIBUTE_ASSOCIATED_CONFIRMATION_VISIBILITY_SCRIPT = "associatedConfirmationVisibilityScript";

    /**
     * @see #setAssociatedScreen(String)
     */
    public static final String ATTRIBUTE_ASSOCIATED_SCREEN = "associatedScreen";

    /**
     * @see #setInstallerTargetScreen(String)
     */
    public static final String ATTRIBUTE_INSTALLER_TARGET_SCREEN = "preferredTargetInstallerScreen";

    /**
     * @see #setUninstallerTargetScreen(String)
     */
    public static final String ATTRIBUTE_UNINSTALLER_TARGET_SCREEN = "preferredTargetUninstallerScreen";

    /**
     * @see #setDefaultMultiExec(boolean)
     */
    public static final String ATTRIBUTE_DEFAULT_MULTI_EXEC = "defaultMultiExec";

    /**
     * @see #setComplementaryStartupLink(boolean)
     */
    public static final String ATTRIBUTE_COMPLEMENTARY_STARTUP_LINK = "complementaryStartupLink";

    /**
     * @see #setUninstallerStartupAction
     */
    public static final String ATTRIBUTE_UNINSTALLER_STARTUP_ACTION = "complementaryStartupAction";

    /**
     * @see #setAssociatedStartupAction(String)
     */
    public static final String ATTRIBUTE_ASSOCIATED_STARTUP_ACTION = "associatedStartupAction";

    /**
     * @see #setDefaultFailureStrategy(FailureStrategy)
     */
    public static final String ATTRIBUTE_DEFAULT_FAILURE_STRATEGY = "defaultFailureStrategy";

    /**
     * @see #setAskUserFailureDefaults(AskUserFailureDefaults)
     */
    public static final String ATTRIBUTE_ASK_USER_FAILURE_DEFAULTS = "askUserFailureDefaults";

    /**
     * @see #setDefaultErrorMessage(String)
     */
    public static final String ATTRIBUTE_DEFAULT_ERROR_MESSAGE = "defaultErrorMessage";

    /**
     * @see #setFullPrivilegesRequired(boolean)
     */
    public static final String ATTRIBUTE_FULL_PRIVILEGES_REQUIRED = "fullPrivilegesRequired";

    /**
     * @see #setWindowsOnly(boolean)
     */
    public static final String ATTRIBUTE_WINDOWS_ONLY = "windowsOnly";

    /**
     * @see #setDefaultActionElevationType
     */
    public static final String ATTRIBUTE_DEFAULT_ACTION_ELEVATION_TYPE = "defaultActionElevationType";

    /**
     * @see #setDefaultActionElevationType
     */
    public static final String ATTRIBUTE_RESTRICT_ACTION_ELEVATION_TYPE = "restrictActionElevationType";

    /**
     * Special identifier for the startup sequence to be used as the {@code className} argument
     * in {@link #setInstallerTargetScreen(String)} and {@link #setUninstallerTargetScreen(String)}.
     */
    public static final String STARTUP_SEQUENCE = "#startup";

    /**
     * @see #setActionListInitializer(ActionListInitializer)
     */
    public static final String ATTRIBUTE_ACTION_LIST_INITIALIZER = "actionListInitializer";


    /**
     * @see #setActionValidator(ActionValidator)
     */
    public static final String ATTRIBUTE_ACTION_VALIDATOR = "actionValidator";

    /**
    /**
     * Same as {@link Install4JBeanInfo#Install4JBeanInfo(String, String, String, boolean, boolean, Integer, Class, Class)}
     */
    protected ActionBeanInfo(String displayName, String shortDescription, String category, boolean multipleInstancesSupported, boolean installedFilesRequired, Integer sortKey, Class<? extends Action> beanClass, Class customizerClass) {
        super(displayName, shortDescription, category, multipleInstancesSupported, installedFilesRequired, sortKey, beanClass, customizerClass);
        init();
    }

    /**
     * Same as {@link #ActionBeanInfo(String, String, String, boolean, boolean, Integer, Class, Class)} with a customizer class of {@code null}.
     */
    protected ActionBeanInfo(String displayName, String shortDescription, String category, boolean multipleInstancesSupported, boolean installedFilesRequired, Integer sortKey, Class<? extends Action> beanClass) {
        super(displayName, shortDescription, category, multipleInstancesSupported, installedFilesRequired, sortKey, beanClass);
        init();
    }

    private void init() {
        if (this instanceof ActionListInitializer) {
            setActionListInitializer((ActionListInitializer)this);
        }
        if (this instanceof ActionValidator) {
            setActionValidator((ActionValidator)this);
        }
    }

    /**
     * Kept for backwards compatibility
     * @deprecated
     * @see #setAssociatedConfirmation(String, String, String)
     */
    @Deprecated
    public void setAssociatedConfirmation(String confirmationDescription, String initScript) {
        setAssociatedConfirmation(confirmationDescription, initScript, null);
    }

    /**
     * Configures an associated confirmation. If the user adds the action, the install4j GUI will ask the user whether to add
     * a question to the associated screen if the screen is a form screen or to the "Additional confirmation" screen
     * otherwise. The form screen will receive a checkbox form component with a variable name equal to the simple
     * class name of the action. The condition expression of the action will be set to {@code context.getBooleanVariable("variableName")}
     * so that the confirmation and the execution of the action are automatically wired together.
     * <p>If the {@code initScript} parameter is not {@code null} and the user accepts the addition of the confirmation,
     * the "Initialization script" property of the checkbox form component is set to {@code initScript}.
     * @param confirmationDescription the message of the confirmation
     * @param initScript the init script, may be {@code null}  @see #ATTRIBUTE_ASSOCIATED_CONFIRMATION
     * @param visibilityScript the visibility script, may be {@code null}  @see #ATTRIBUTE_ASSOCIATED_CONFIRMATION
     * @see #ATTRIBUTE_ASSOCIATED_CONFIRMATION_INIT_SCRIPT
     */
    public void setAssociatedConfirmation(String confirmationDescription, String initScript, String visibilityScript) {
        if (confirmationDescription != null) {
            getBeanDescriptor().setValue(ATTRIBUTE_ASSOCIATED_CONFIRMATION, confirmationDescription);
        }
        if (initScript != null) {
            getBeanDescriptor().setValue(ATTRIBUTE_ASSOCIATED_CONFIRMATION_INIT_SCRIPT, initScript);
        }
        if (visibilityScript != null) {
            getBeanDescriptor().setValue(ATTRIBUTE_ASSOCIATED_CONFIRMATION_VISIBILITY_SCRIPT, visibilityScript);
        }
    }

    /**
     * Configures an associated screen. If set, and the user adds the action, the install4j GUI checks if
     * an instance of the specified class is present in the list of configured screens. If not, it will ask the user whether to add
     * such a screen.
     * @param className the class name of the associated screen
     * @see #ATTRIBUTE_ASSOCIATED_SCREEN
     */
    public void setAssociatedScreen(String className) {
        if (className != null) {
            getBeanDescriptor().setValue(ATTRIBUTE_ASSOCIATED_SCREEN, className);
        }
    }

    /**
     * Configures a target screen for the installer. If set, and the user adds the action, the install4j GUI checks
     * if the current screen is an instance of the specified class. If not, it will ask the user whether the action
     * should rather be added to the target screen, adding that screen if it is not present in the list of configured screens.
     * @param className the class name of the target screen or {@link #STARTUP_SEQUENCE} for the startup sequence
     * @see #ATTRIBUTE_INSTALLER_TARGET_SCREEN
     */
    public void setInstallerTargetScreen(String className) {
        if (className != null) {
            getBeanDescriptor().setValue(ATTRIBUTE_INSTALLER_TARGET_SCREEN, className);
        }
    }

    /**
     * Configures a target screen for the uninstaller. If set, and the user adds the action, the install4j GUI checks
     * if the current screen is an instance of the specified class. If not, it will ask the user whether the action
     * should rather be added to the target screen, adding that screen if it is not present in the list of configured screens.
     * @param className the class name of the target screen or {@link #STARTUP_SEQUENCE} for the startup sequence
     * @see #ATTRIBUTE_UNINSTALLER_TARGET_SCREEN
     */
    public void setUninstallerTargetScreen(String className) {
        if (className != null) {
            getBeanDescriptor().setValue(ATTRIBUTE_UNINSTALLER_TARGET_SCREEN, className);
        }
    }

    /**
     * Configures the default value for the "Can be executed multiple times" property of the action.
     * @param multiExec the default value
     * @see #ATTRIBUTE_DEFAULT_MULTI_EXEC
     */
    public void setDefaultMultiExec(boolean multiExec) {
        getBeanDescriptor().setValue(ATTRIBUTE_DEFAULT_MULTI_EXEC, multiExec);
    }

    /**
     * Determines if a complementary startup link should be placed in the installer or uninstaller.
     * If the action is added to the startup sequence of the installer, the user will be asked whether a link to the action
     * should be placed in the startup sequence of the uninstaller or vice versa. This setting has no effect if the action
     * is added to any other screen.
     * @param complementaryStartupLink the value
     * @see #ATTRIBUTE_COMPLEMENTARY_STARTUP_LINK
     */
    public void setComplementaryStartupLink(boolean complementaryStartupLink) {
        getBeanDescriptor().setValue(ATTRIBUTE_COMPLEMENTARY_STARTUP_LINK, complementaryStartupLink);
    }

    /**
     * Configures an associated startup action. If set, and the user adds the action, the install4j GUI checks
     * if the startup sequence contains an action of the specified class. If not, it will ask the user whether the action
     * of that class should be added to the startup sequence.
     * @param className the class name of the associated startup action
     * @see #ATTRIBUTE_ASSOCIATED_STARTUP_ACTION
     */
    public void setAssociatedStartupAction(String className) {
        if (className != null) {
            getBeanDescriptor().setValue(ATTRIBUTE_ASSOCIATED_STARTUP_ACTION, className);
        }
    }

    /**
     * Determines if a complementary startup action should be placed in the uninstaller.
     * If the action is added to the startup sequence of the installer, the user will be asked whether a link to the action
     * should be placed in the startup sequence of the uninstaller. This setting has no effect if the action
     * is added to any other screen.
     * @param className the class name of the complementary startup action
     * @see #ATTRIBUTE_UNINSTALLER_STARTUP_ACTION
     */
    public void setUninstallerStartupAction(String className) {
        if (className != null) {
            getBeanDescriptor().setValue(ATTRIBUTE_UNINSTALLER_STARTUP_ACTION, className);
        }
    }

    /**
     * Configures the default failure strategy for the action.
     * @param failureStrategy the default failure strategy
     * @see #ATTRIBUTE_DEFAULT_FAILURE_STRATEGY
     */
    public void setDefaultFailureStrategy(FailureStrategy failureStrategy) {
        if (failureStrategy != null) {
            getBeanDescriptor().setValue(ATTRIBUTE_DEFAULT_FAILURE_STRATEGY, failureStrategy);
        }
    }

    /**
     * Configures the questions that are selected by default if the {@link FailureStrategy#ASK_USER} failure
     * strategy is selected.
     * @param askUserFailureDefaults the default questions that are selected
     * @see #ATTRIBUTE_ASK_USER_FAILURE_DEFAULTS
     */
    public void setAskUserFailureDefaults(AskUserFailureDefaults askUserFailureDefaults) {
        if (askUserFailureDefaults != null) {
            getBeanDescriptor().setValue(ATTRIBUTE_ASK_USER_FAILURE_DEFAULTS, askUserFailureDefaults);
        }
    }

    /**
     * Configures the default error message for the action.
     * @param errorMessage the default error message
     * @see #ATTRIBUTE_DEFAULT_ERROR_MESSAGE
     */
    public void setDefaultErrorMessage(String errorMessage) {
        if (errorMessage != null) {
            getBeanDescriptor().setValue(ATTRIBUTE_DEFAULT_ERROR_MESSAGE, errorMessage);
        }
    }

    /**
     * Configures if full privileges are required or not. If this method is called with the argument
     * {@code true}, the install4j IDE will ask the user if all the failure properties of the "Request privileges"
     * action in the startup sequence should be selected. If that action does not exist, it will be created.
     * <p>
     *     Note: Setting this property to {@code true} automatically implies {@link #setDefaultActionElevationType}}
     *      set to {@link ActionElevationType#ELEVATE}.
     * </p>
     * @param fullPrivilegesRequired the value
     */
    public void setFullPrivilegesRequired(boolean fullPrivilegesRequired) {
        BeanDescriptor beanDescriptor = getBeanDescriptor();
        beanDescriptor.setValue(ATTRIBUTE_FULL_PRIVILEGES_REQUIRED, fullPrivilegesRequired);
        beanDescriptor.setValue(ATTRIBUTE_RESTRICT_ACTION_ELEVATION_TYPE, Boolean.TRUE);
    }

    /**
     * Same as {@link #setDefaultActionElevationType(ActionElevationType, boolean)} with {@code restrict} set to
     * {@code true}.
     * @param actionElevationType the elevation type
     */
    public void setDefaultActionElevationType(ActionElevationType actionElevationType) {
        getBeanDescriptor().setValue(ATTRIBUTE_DEFAULT_ACTION_ELEVATION_TYPE, actionElevationType);
    }

    /**
     * Configures in which elevation mode the action should run in elevated mode by default. If not set,
     * the elevation mode will be inherited from the parent element. If the {@code restrict} parameter is set to
     * {@code false}, the user can always change the elevation type in the install4j IDE. Otherwise, the default
     * is fixed.
     * @param actionElevationType the elevation type
     * @param restrict whether the default setting should be the only allowed setting
     */
    public void setDefaultActionElevationType(ActionElevationType actionElevationType, boolean restrict) {
        BeanDescriptor beanDescriptor = getBeanDescriptor();
        beanDescriptor.setValue(ATTRIBUTE_DEFAULT_ACTION_ELEVATION_TYPE, actionElevationType);
        beanDescriptor.setValue(ATTRIBUTE_RESTRICT_ACTION_ELEVATION_TYPE, restrict);
    }

    /**
     * Configures if this action is a Windows-only action. The install4j IDE will take this into account when changing
     * properties of the "Request privileges" action based on the value set with {@link #setFullPrivilegesRequired}.
     * @param windowsOnly the value
     */
    public void setWindowsOnly(boolean windowsOnly) {
        BeanDescriptor beanDescriptor = getBeanDescriptor();
        beanDescriptor.setValue(ATTRIBUTE_WINDOWS_ONLY, windowsOnly);
    }

    /**
     * Configures an action list initializer.
     * @param actionListInitializer the bean initializer.
     * @see ActionListInitializer
     * @see #ATTRIBUTE_ACTION_LIST_INITIALIZER
     */
    public void setActionListInitializer(ActionListInitializer actionListInitializer) {
        if (actionListInitializer != null) {
            getBeanDescriptor().setValue(ATTRIBUTE_ACTION_LIST_INITIALIZER, actionListInitializer);
        }
    }

    /**
     * Configures an action validator.
     * @param actionValidator the action validator.
     * @see ActionValidator
     * @see #ATTRIBUTE_ACTION_VALIDATOR
     */
    public void setActionValidator(ActionValidator actionValidator) {
        if (actionValidator != null) {
            getBeanDescriptor().setValue(ATTRIBUTE_ACTION_VALIDATOR, actionValidator);
        }
    }


}
