package com.install4j.api.unix;

import com.ejt.framework.util.CommonStringUtil;
import com.install4j.api.Util;
import com.install4j.runtime.installer.helper.FileAttributesHelper;
import com.install4j.runtime.installer.helper.Logger;
import com.install4j.runtime.installer.platform.unix.Execution;
import com.install4j.runtime.installer.platform.unix.LegacyUnixFileSystem;

import java.io.File;
import java.io.IOException;
import java.nio.file.*;
import java.nio.file.attribute.*;

/**
 * Collection of static methods to access Unix-specific file operations and information.
 * @author ej-technologies GmbH
 */
public class UnixFileSystem {

    /**
     * Set the Unix file mode for a file. For specifying integer file modes, you can use octal numbers by prefixing
     * a "0".
     * @param intMode the Unix file mode
     * @param destFile the file for which the mode should be set
     * @return whether the operation was successful or not
     */
    public static boolean setMode(int intMode, File destFile) {
        if (!Util.isWindows()) {
            try {
                setPosixFilePermissionsInternal(destFile, CommonStringUtil.toJavaTypeMode(intMode));
            } catch (Exception e) {
                return false;
            }
        }
        return true;
    }

    /**
     * Set the Unix file mode for a file.
     * @param mode the Unix file mode as an octal string, e.g. "755".
     * @param destFile the file for which the mode should be set
     * @return whether the operation was successful or not
     */
    public static boolean setMode(String mode, File destFile) {
        if (!Util.isWindows()) {
            try {
                if (mode != null && !mode.isEmpty()) {
                    setPosixFilePermissionsInternal(destFile, FileAttributesHelper.translateMode(mode, destFile));
                }
            } catch (Exception e) {
                Logger.getInstance().log(null, "set permissions: " + e, false);
                return false;
            }
        }
        return true;
    }

    private static void setPosixFilePermissionsInternal(File file, String mode) throws IOException {
        try {
            Path path = file.toPath();
            try {
                Files.setPosixFilePermissions(path, PosixFilePermissions.fromString(mode));
            } catch (IOException e) {
                if (!Files.isSymbolicLink(path)) {
                    throw e;
                }
            }
        } catch (InvalidPathException e) {
            throw new IOException(e);
        }
    }


    /**
     * Create a symbolic link on Unix. If the link exists, it will be overwritten.
     * @param destFileName the file where the link should point to
     * @param linkFile the link that should be created
     * @return whether the operation was successful or not
     */
    public static boolean createLink(String destFileName, File linkFile) {
        if (!Util.isWindows()) {
            linkFile.delete();
            try {
                Files.createSymbolicLink(linkFile.toPath(), Paths.get(destFileName));
                return true;
            } catch (Throwable e) {
                Logger.getInstance().log(null, "could not create symbolic link: " + e.getMessage(), false);
            }
        }
        return false;
    }

    /**
     * Set the owner of a file.
     * @param owner the owner spec. Either a user name like "user" or a user name followed by a colon and a group name, like "user:group".
     *              In the latter case, "user" can be the empty string and only the group ownership will be changed.
     * @param file the file for which the owner should be set.
     * @return whether the operation was successful or not
     */
    public static boolean setOwner(String owner, File file) {
        try {
            int colonPos = owner.indexOf(':');
            String user;
            String group = "";
            if (colonPos > -1) {
                user = owner.substring(0, colonPos);
                group = owner.substring(colonPos + 1);
            } else {
                user = owner;
            }
            PosixFileAttributeView posixFileAttributeView = Files.getFileAttributeView(file.toPath(), PosixFileAttributeView.class, LinkOption.NOFOLLOW_LINKS);

            UserPrincipalLookupService principalLookupService = FileSystems.getDefault().getUserPrincipalLookupService();
            if (!user.isEmpty()) {
                UserPrincipal userPrincipal = principalLookupService.lookupPrincipalByName(user);
                posixFileAttributeView.setOwner(userPrincipal);
            }
            if (!group.isEmpty()) {
                GroupPrincipal groupPrincipal = principalLookupService.lookupPrincipalByGroupName(group);
                posixFileAttributeView.setGroup(groupPrincipal);
            }
        } catch (Exception e) {
            Logger.getInstance().log(null, "set owner: " + e, false);
            return false;
        }
        return true;
    }

    /**
     * Get Unix-specific information about a file.
     * @param file the file for which the information is requested.
     * @return the {@code FileInformation} object with the Unix-specific information
     */
    public static FileInformation getFileInformation(File file) {
        try {
            PosixFileAttributes attributes = FileAttributesHelper.getFileAttributes(file);
            int mode = LegacyUnixFileSystem.fromJavaTypeMode(PosixFilePermissions.toString(attributes.permissions()));
            String ownerInfo = attributes.owner().getName() + ":" + attributes.group().getName();
            String linkTarget = null;
            if (attributes.isSymbolicLink()) {
                linkTarget = Files.readSymbolicLink(file.toPath()).toString();
            }
            return new LegacyUnixFileSystem.FileInformationImpl(mode, ownerInfo, attributes.isSymbolicLink(), linkTarget);
        } catch (Exception e) {
            Logger.getInstance().log(e);
            return null;
        }
    }

    /**
     * Returns the absolute path of an executable if it is located on the path.
     * @param executable the executable name without path information or quotes
     * @return the path of the executable or {@code null} if the executable was not found.
     */
    public static File findExecutableInPath(String executable) {
        StringBuffer output = new StringBuffer();
        try {
            if (!Execution.executeAndWait(new String[] { getShell(), "-c", "command -v " + executable}, output) || output.length() == 0) {
                return null;
            } else {
                return new File(toFirstNewLine(output.toString()));
            }
        } catch (Exception e) {
            Logger.getInstance().log(e);
            return null;
        }
    }

    /**
     * Returns the shell executable of the current user.
     * If the {@code SHELL} environment variable is set, its value is returned, otherwise the return value is
     * {@code /bin/sh}
     */
    public static String getShell() {
        String shell = System.getenv("SHELL");
        if (shell != null) {
            return shell;
        } else {
            return "/bin/sh";
        }
    }

    private static String toFirstNewLine(String pathname) {
        int newLineIndex = pathname.indexOf('\n');
        if (newLineIndex > -1) {
            return pathname.substring(0, newLineIndex);
        } else {
            return pathname;
        }
    }

    /**
     * Holds Unix-specific information about a file.
     */
    public interface FileInformation {
        /**
         * Get the Unix mode of a file. To convert this mode to a human-readable octal string, use
         * {@code Integer.toOctalString}.
         * @return the Unix mode
         */
        int getMode();

        /**
         * Get the owner of a file. Depending on the platform, this is either a user name like
         * "user" or a user name followed by a colon and a group name, like "user:group".
         * @return the owner of the file.
         */
        String getOwnerInfo();

        /**
         * Return whether the file is a link or not.
         * @return {@code true or false}.
         */
        boolean isLink();

        /**
         * If {@code isLink} returns {@code true}, this method returns the target file of the link.
         * @return the target file if the file is a link or {@code null} otherwise.
         * @see #isLink()
         */
        String getLinkTarget();
    }

}
