package com.install4j.api.events;

import com.install4j.api.context.Context;

import java.io.File;

/**
 * The event object that is passed to installer event listeners for single file installation events types.
 * Event listeners are registered in the {@link com.install4j.api.context.InstallerContext}.
 * <p>This event object contains additional information about the installed file.
 * The event types {@link EventType#BEFORE_INSTALL_FILE} and {@link EventType#AFTER_INSTALL_FILE}
 * produce this event object when fired.
 * @author ej-technologies GmbH
 * @see InstallerEventListener
 * @see com.install4j.api.context.InstallerContext#addInstallerEventListener(InstallerEventListener)
 *
 */
public class InstallerFileEvent extends InstallerEvent {
    private File destinationFile;
    private long fileSize;

    /**
     * The constructor is called by the framework.
     * @param source the event source
     * @param context the context
     * @param type the event type.
     * @param success whether the operation associated with the event was successful
     * @param destinationFile the installed file
     * @param fileSize the size of the installed file
     */
    public InstallerFileEvent(Object source, Context context, EventType type, boolean success, File destinationFile, long fileSize) {
        super(source, context, type, success);
        this.destinationFile = destinationFile;
        this.fileSize = fileSize;
    }

    /**
     * The constructor is called by the framework.
     * @param source the event source
     * @param context the context
     * @param type the event type.
     * @param destinationFile the installed file
     * @param fileSize the size of the installed file
     */
    public InstallerFileEvent(Object source, Context context, EventType type, File destinationFile, long fileSize) {
        super(source, context, type);
        this.destinationFile = destinationFile;
        this.fileSize = fileSize;
    }

    /**
     * Get the installed file location associated with this event.
     * @return the file
     */
    public File getDestinationFile() {
        return destinationFile;
    }

    /**
     * Get the size of the installed file in bytes.
     * @return the size in bytes
     */
    public long getFileSize() {
        return fileSize;
    }

    @Override
    public String toString() {
        return super.toString() + " [file " + destinationFile + ", size " + fileSize +  "]";
    }

    @Override
    public String getVerbose() {
        String ret = super.getVerbose() + ": " + destinationFile;
        if (getType() == EventType.BEFORE_INSTALL_FILE) {
            ret += "; size: " + fileSize + " bytes; exists: " + destinationFile.exists();
        }
        return ret;
    }
}
