package com.install4j.api.beaninfo;

import java.beans.IntrospectionException;

/**
 * Property descriptor for properties of type {@link com.install4j.api.beans.ScriptProperty}.
 * With this descriptor you define the return type and the parameters that are passed to the script by
 * {@link com.install4j.api.context.Context#runScript(com.install4j.api.beans.ScriptProperty, com.install4j.api.beans.Bean, Object[])}
 * at runtime.
 * <p>Using this class is not strictly required. In principle, you could also set values for the {@code ATTRIBUTE_*} constants in the property descriptor
 * of an unrelated property descriptor class.
 * @see ScriptParameter
 */
public class ScriptPropertyDescriptor extends Install4JPropertyDescriptor {

    /**
     * @see #create(String, Class, String, String, Class, ScriptParameter[])
     */
    public static final String ATTRIBUTE_RETURN_TYPE = "returnType";

    /**
     * @see #create(String, Class, String, String, Class, ScriptParameter[])
     */
    public static final String ATTRIBUTE_SCRIPT_PARAMETERS = "scriptParameters";

    /**
     * @see #setCodeGalleryTags(String[])
     */
    public static final String ATTRIBUTE_CODE_GALLERY_TAGS = "codeGalleryTags";

    /**
     * Create a file property descriptor as with {@link Install4JPropertyDescriptor#create(String, Class, String, String)}
     * and additional configuration specific to script properties.
     * @param returnType the class or the return type. Classes for primitive types are written as {@code void.class} or {@code int.class}.
     * @param scriptParameters the additional parameters for the script or {@code null} if no additional parameters are passed.
     * @return the property descriptor
     * @see com.install4j.api.beans.ScriptProperty
     * @see com.install4j.api.context.Context#runScript(com.install4j.api.beans.ScriptProperty, com.install4j.api.beans.Bean, Object[])
     */
    public static ScriptPropertyDescriptor create(String propertyName, Class beanClass, String displayName, String shortDescription, Class returnType, ScriptParameter[] scriptParameters) {
        try {
            ScriptPropertyDescriptor descriptor = ScriptPropertyDescriptor.createDescriptor(propertyName, beanClass);
            descriptor.setDisplayName(displayName);
            descriptor.setShortDescription(shortDescription);
            descriptor.setReturnType(returnType);
            descriptor.setScriptParameters(scriptParameters);
            return descriptor;
        } catch (IntrospectionException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Sets the code gallery tags that are used to build the code gallery for this script property.
     * This is only relevant if you supply your own code gallery by including a code gallery file (see
     * {@code $INSTALL4J_HOME/resource/codeGallery.dtd}) in the extension manifest (attribute {@code Code-Gallery: true})
     * @param tags the additional tags that should be included in the code gallery
     * @return {@code this}, for chained calls on this property descriptor
     */
    public ScriptPropertyDescriptor setCodeGalleryTags(String[] tags) {
        if (tags != null) {
            setValue(ScriptPropertyDescriptor.ATTRIBUTE_CODE_GALLERY_TAGS, tags);
        }
        return this;
    }

    private void setReturnType(Class returnType) {
        if (returnType != null) {
            setValue(ScriptPropertyDescriptor.ATTRIBUTE_RETURN_TYPE, returnType);
        }
    }

    private void setScriptParameters(ScriptParameter[] scriptParameters) {
        if (scriptParameters != null) {
            setValue(ScriptPropertyDescriptor.ATTRIBUTE_SCRIPT_PARAMETERS, scriptParameters);
        }
    }

    private static ScriptPropertyDescriptor createDescriptor(String propertyName, Class beanClass) throws IntrospectionException {
        try {
            return new ScriptPropertyDescriptor(propertyName, beanClass);
        } catch (Exception ignored) {
        }
        return new ScriptPropertyDescriptor(propertyName, beanClass, "get" + capitalize(propertyName), null);
    }

    ScriptPropertyDescriptor(String propertyName, Class beanClass) throws IntrospectionException {
        super(propertyName, beanClass);
    }

    ScriptPropertyDescriptor(String propertyName, Class beanClass, String readMethod, String writeMethod) throws IntrospectionException {
        super(propertyName, beanClass, readMethod, writeMethod);
    }

}
