package com.install4j.api.windows;

import com.install4j.api.Util;
import com.install4j.runtime.installer.helper.Logger;
import com.install4j.runtime.installer.platform.win32.Misc;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.LockSupport;

/**
 * Collection of methods to retrieve information about windows
 * on Microsoft Windows.
 * @author ej-technologies GmbH
 */
public class WindowInfo {

    private final long hwnd;
    private final int style;
    private final String className;
    private final String windowTitle;

    private static final int WS_MAXIMIZEBOX = 0x00010000;

    private WindowInfo(long hwnd, int style, String className, String windowTitle) {
        this.hwnd = hwnd;
        this.style = style;
        this.className = className;
        this.windowTitle = windowTitle;
    }

    /**
     * Returns all visible top level windows of the process with the supplied process id.
     * @param processId the process id
     * @return the top level windows or an empty list if the process does not exist.
     */
    public static Collection<WindowInfo> getTopLevelWindows(long processId) {
        return Misc.getTopLevelWindows((int)processId);
    }

    /**
     * Checks if Windows 10 is currently used in tablet mode.
     * @return {@code true} if tablet mode is activated
     */
    public static boolean isTabletMode() {
        return Misc.isTabletMode();
    }

    /**
     * Wait until a process with the given path displays a maximizable window. When Windows is in tablet mode,
     * you can use this method to delay the termination of an update downloader until a subsequently started installer can be displayed in full-screen mode.
     * @param executablePath the executable path
     * @param timeout a timeout value or 0 if no timeout should occur
     * @param unit the unit of the timeout value
     * @return {@code true} if a process with the given executable path was found that displays a maximizable window within the timeout period, {@code false} otherwise.
     */
    public static boolean waitForMaximizableWindow(File executablePath, long timeout, TimeUnit unit) {
        if (!Util.isWindows()) {
            return false;
        }
        try {
            long timeoutNanos = unit.toNanos(timeout);
            long startTime = System.nanoTime();
            File canonicalPath = executablePath.getCanonicalFile();
            List<Integer> pids = getPids(canonicalPath, startTime, timeoutNanos);
            if (pids.isEmpty()) {
                return false;
            }
            while (true) {
                for (Integer pid : pids) {
                    for (WindowInfo windowInfo : getTopLevelWindows(pid)) {
                        if ((windowInfo.getStyle() & WS_MAXIMIZEBOX) > 0) {
                            return true;
                        }
                    }
                }
                if (isTimeout(startTime, timeoutNanos)) {
                    return false;
                }
                LockSupport.parkNanos(1000 * 1000 * 100);
            }
        } catch (IOException e) {
            Logger.getInstance().log(e);
            return false;
        }
    }

    private static List<Integer> getPids(File canonicalPath, long startTime, long timeoutNanos) {
        List<Integer> pids = new ArrayList<>();
        while (true) {
            for (WinProcesses.Info info : WinProcesses.getRunningProcesses()) {
                try {
                    if (new File(info.getModuleName()).getCanonicalFile().equals(canonicalPath)) {
                        pids.add(info.getProcessId());
                    }
                } catch (IOException e) {
                    //continue
                }
            }
            if (!pids.isEmpty() || isTimeout(startTime, timeoutNanos)) {
                return pids;
            }
            LockSupport.parkNanos(1000 * 1000 * 100);
        } 
    }

    private static boolean isTimeout(long startTime, long timeoutNanos) {
        return timeoutNanos > 0 && (System.nanoTime() - startTime) > timeoutNanos;
    }


    /**
     * Returns the window handle.
     * @return the window handle
     */
    public long getHwnd() {
        return hwnd;
    }

    /**
     * Returns the window style. This is the style int retrieved with {@code GetWindowLongPtr} and {@code GWL_STYLE}.
     * @return the style
     */
    public int getStyle() {
        return style;
    }

    /**
     * Returns the window class name.
     * @return the class name
     */
    public String getClassName() {
        return className;
    }

    /**
     * Returns the window title.
     * @return the title
     */
    public String getWindowTitle() {
        return windowTitle;
    }

    @Override
    public String toString() {
        return "WindowInfo{" +
                "hwnd=" + hwnd +
                ", style=0x" + Integer.toHexString(style) +
                ", className='" + className + '\'' +
                ", windowTitle='" + windowTitle + '\'' +
                '}';
    }
}
