package com.install4j.api.update;

import java.io.Serializable;
import java.net.URL;

/**
 * Contains information about a single media file available for update. Instances of this
 * class are returned by an {@link UpdateDescriptor}.
 *
 * @author ej-technologies GmbH
 */
public interface UpdateDescriptorEntry extends Serializable {
    /**
     * Returns the media file ID for which this update is suitable.
     * <p>
     * Note: install4j creates one update descriptor entry for
     * each media file. If you discontinue a media file, you can duplicate a suitable update descriptor entry in
     * {@code updates.xml} and set its target media file ID to that of the discontinued media file.
     * @return the media file ID
     */
    String getTargetMediaFileId();

    /**
     * Returns the minimum version that is eligible for an update.
     * <p>
     * By default, this is the empty string. If you want to set a minimum version, you can
     * change the {@code updatableVersionMin} attribute on the corresponding {@code entry} element
     * in the {@code updates.xml} file.
     * @return the minimum version
     */
    String getUpdatableVersionMin();

    /**
     * Returns the maximum version that is eligible for an update.
     * <p>
     * By default, this is the empty string. If you want to set a maximum version, you can
     * change the {@code updatableVersionMax} attribute on the corresponding {@code entry} element
     * in the {@code updates.xml} file.
     * @return the maximum version
     */
    String getUpdatableVersionMax();

    /**
     * Returns the file name of the update installer.
     * @return the file name
     */
    String getFileName();

    /**
     * Returns the full URL from which the update installer can be downloaded.
     * This includes the file name and is based on the {@link UpdateDescriptor#getBaseUrl() base url}
     * of the {@link UpdateDescriptor update descriptor}.
     * @return the URL
     */
    URL getURL();

    /**
     * Returns the version of the available update.
     * @return the new version
     */
    String getNewVersion();

    /**
     * Returns the media file ID of the update installer that is represented by this entry.
     * @return the media file ID
     */
    String getNewMediaFileId();

    /**
     * Returns whether the media file is an installer or an archive.
     * @return {@code true} for an archive
     */
    boolean isArchive();

    /**
     * Returns whether the media file is a macOS single bundle archive.
     * @return {@code true} for a single bundle archive
     */
    boolean isSingleBundle();

    /**
     * Returns the file size of the update installer in bytes. The size is determined at compile time.
     * @return the file size in bytes
     */
    long getFileSize();

    /**
     * Returns the SHA-256 sum of the file if the option to calculate checksums has been enabled. That option is
     * found on the General Settings-&gt;Media File Options step in the install4j IDE. If the option is disabled,
     * this method returns null.
     * @return the SHA-256 sum as a hex string or null if checksums are not calculated
     */
    String getSha256Sum();

    /**
     * Returns the MD5 sum of the file if the option to calculate checksums has been enabled. That option is
     * found on the General Settings-&gt;Media File Options step in the install4j IDE. If the option is disabled,
     * this method returns null.
     * @return the MD5 sum as a hex string or null if checksums are not calculated
     */
    String getMd5Sum();

    /**
     * Returns the file size of the update installer as a verbose string. The size is determined at compile time.
     * @return the file size as a string
     */
    String getFileSizeVerbose();

    /** Returns the file name of the bundled JRE. If no JRE is bundled, this is the empty string.
     * @return the file name of the bundled JRE or the empty string
     */
    String getBundledJre();

    /** Returns the required minimum version of the JRE.
     * @return the minimum version or the empty string
     */
    String getJreMinVersion();

    /** Returns the required maximum version of the JRE.
     * @return the maximum version or the empty string
     */
    String getJreMaxVersion();

    /**
     * Returns the comment for this entry. By default, the comment is empty. You can add
     * comments in the {@code comment} elements in the {@code updates.xml} file. The comment with the language that matches
     * the current locale is returned. If there is no matching language, the comment for the principal language is returned.
     * <p>The default standalone update downloader offers the user a hyperlink to display this comment for an available update.
     * @return the comment
     */
    String getComment();

    /**
     * Returns an additional attribute that you have added to the {@code entry} element in the {@code updates.xml} file
     * represented by this instance.
     * @param attributeName the name of the additional attribute
     * @param defaultValue the default value to be returned if the attribute does not exist
     * @return the attribute value
     */
    String getAdditionalAttribute(String attributeName, String defaultValue);

    /**
     * Checks if a version is compatible with the minimum and maximum updatable versions and lower than the new version
     * of this update descriptor entry. In that case, this update descriptor entry is a candidate for downloading an
     * update when only considering version numbers.
     * @param version the version
     * @return {@code true} if compatible
     */
    boolean checkVersionCompatible(String version);
    
    /**
     * Checks if an updater with this version is already downloaded and scheduled for installation by a "Schedule update installation" action.
     * @return {@code true} if scheduled
     */
    boolean isDownloaded();
}
