package com.install4j.api.events;

import com.install4j.api.context.Context;

import java.util.List;

/**
 * The event object that is passed to installer event listeners after an action has been executed.
 * Event listeners are registered in the {@link com.install4j.api.context.InstallerContext}.
 * <p>This event object contains additional information about the duration of the action.
 * @author ej-technologies GmbH
 * @see InstallerEventListener
 * @see com.install4j.api.context.InstallerContext#addInstallerEventListener(InstallerEventListener)
 *
 */
public class InstallerActionEvent extends InstallerEvent {

    private final String actionId;
    private long duration;
    private Exception exception;
    private List<String> log;

    /**
     * The constructor is called by the framework.
     * @param source the event source
     * @param context the context
     * @param type the event type.
     * @param success whether the operation associated with the event was successful
     * @param actionId the ID of the action
     * @param duration the duration in milliseconds
     * @param exception the exception if the action terminated abnormally or {@code null}
     * @param log the log output produced by the action
     */
    public InstallerActionEvent(Object source, Context context, EventType type, boolean success, String actionId, long duration, Exception exception, List<String> log) {
        super(source, context, type, success);
        this.actionId = actionId;
        this.duration = duration;
        this.exception = exception;
        this.log = log;
    }

    /**
     * Returns the ID of the action.
     * @return the ID
     */
    public String getActionId() {
        return actionId;
    }

    /**
     * The exception if the action terminated abnormally or {@code null}.
     * @return the ID
     */
    public Exception getException() {
        return exception;
    }

    /**
     * Returns the log output produced by the action
     * @return a list of single log outputs
     */
    public List<String> getLog() {
        return log;
    }

    /**
     * Returns the duration of the action in milliseconds
     * @return the duration in milliseconds
     */
    public long getDuration() {
        return duration;
    }

    @Override
    public String toString() {
        return super.toString() + "[id " + actionId + "]" + "[duration " + duration + " ms]";
    }

    @Override
    public String getVerbose() {
        return super.getVerbose() + " after " + duration + " ms";
    }
}
