package com.install4j.api.context;

import com.install4j.api.styles.StyleManager;

/**
 * The wizard context allows you to modify the GUI installer or uninstaller. It is returned by
 * {@link Context#getWizardContext()} and is only functional if the installer or uninstaller
 * is running in GUI mode.
 * @author ej-technologies GmbH
 * @see Context#getWizardContext()
 */
public interface WizardContext {

    /**
     * Change the enabled status of control buttons on the current screen.
     * This setting applies only to the currently displayed screen.
     * @param controlButtonType the type of the control buttons
     * @param enabled if the buttons should be enabled or not
     */
    void setControlButtonEnabled(ControlButtonType controlButtonType, boolean enabled);

    /**
     * Change the visibility of control buttons on the current screen.
     * This setting applies only to the currently displayed screen.
     * @param controlButtonType the type of the control buttons
     * @param visible if the buttons should be visible or not
     */
    void setControlButtonVisible(ControlButtonType controlButtonType, boolean visible);

    /**
     * Perform the action associated with a particular control button.
     * @param controlButtonType the type of the control button
     */
    void pressControlButton(ControlButtonType controlButtonType);

    /**
     * Transfer the focus to a particular control button.
     * If there are multiple control buttons of the specified type, the first one will be used.
     * @param controlButtonType the type of the control button
     */
    void focusControlButton(ControlButtonType controlButtonType);

    /**
     * Set the test of control buttons on the current screen.
     * This setting applies only to the currently displayed screen.
     * @param controlButtonType the type of the control buttons
     * @param text the new text
     */
    void setControlButtonText(ControlButtonType controlButtonType, String text);

    /**
     * Change the enabled status of the "Next" button at the bottom of the current screen.
     * This setting applies only to the currently displayed screen.
     * @param enabled if the button should be enabled or not
     * @deprecated use {@link #setControlButtonEnabled(ControlButtonType, boolean)} with {@link ControlButtonType#NEXT} instead
     */
    @Deprecated
    void setNextButtonEnabled(boolean enabled);

    /**
     * Change the enabled status of the "Back" button at the bottom of the current screen.
     * This setting applies only to the currently displayed screen.
     * @param enabled if the button should be enabled or not
     * @deprecated use {@link #setControlButtonEnabled(ControlButtonType, boolean)} with {@link ControlButtonType#PREVIOUS} instead
     */
    @Deprecated
    void setBackButtonEnabled(boolean enabled);

    /**
     * Change the enabled status of the "Cancel" button at the bottom of the current screen.
     * This setting applies only to the currently displayed screen.
     * @param enabled if the button should be enabled or not
     * @deprecated use {@link #setControlButtonEnabled(ControlButtonType, boolean)} with {@link ControlButtonType#CANCEL} instead
     */
    @Deprecated
    void setCancelButtonEnabled(boolean enabled);

    /**
     * Change the visibility of the "Next" button at the bottom of the current screen.
     * This setting applies only to the currently displayed screen.
     * @param visible if the button should be visible or not
     * @deprecated use {@link #setControlButtonVisible(ControlButtonType, boolean)} with {@link ControlButtonType#NEXT} instead
     */
    @Deprecated
    void setNextButtonVisible(boolean visible);

    /**
     * Change the visibility of the "Back" button at the bottom of the current screen.
     * This setting applies only to the currently displayed screen.
     * @param visible if the button should be visible or not
     * @deprecated use {@link #setControlButtonVisible(ControlButtonType, boolean)} with {@link ControlButtonType#PREVIOUS} instead
     */
    @Deprecated
    void setBackButtonVisible(boolean visible);

    /**
     * Change the visibility of the "Cancel" button at the bottom of the current screen.
     * This setting applies only to the currently displayed screen.
     * @param visible if the button should be visible or not
     * @deprecated use {@link #setControlButtonVisible(ControlButtonType, boolean)} with {@link ControlButtonType#CANCEL} instead
     */
    @Deprecated
    void setCancelButtonVisible(boolean visible);

    /**
     * Move to the next screen. This has the same effect as if the user clicks on the Next button.
     * @deprecated use {@link #pressControlButton(ControlButtonType)} with {@link ControlButtonType#NEXT} instead
     */
    @Deprecated
    void pressNextButton();

    /**
     * Move to the previous screen. This has the same effect as if the user clicks on the Back button.
     * @deprecated use {@link #pressControlButton(ControlButtonType)} with {@link ControlButtonType#PREVIOUS} instead
     */
    @Deprecated
    void pressPreviousButton();

    /**
     * Cancel the installer or uninstaller. This has the same effect as if the user clicks on the Cancel button.
     * @deprecated use {@link #pressControlButton(ControlButtonType)} with {@link ControlButtonType#CANCEL} instead
     */
    @Deprecated
    void pressCancelButton();

    /**
     * Transfer the focus to the Next button. This could be called in the "Post-activation" script of a screen.
     * @deprecated use {@link #focusControlButton(ControlButtonType)} with {@link ControlButtonType#NEXT} instead
     */
    @Deprecated
    void focusNextButton();

    /**
     * Set the text on the "Next" button.
     * @param text the new text
     * @deprecated use {@link #setControlButtonText(ControlButtonType, String)} with {@link ControlButtonType#NEXT} instead
     */
    @Deprecated
    void setNextButtonText(String text);

    /**
     * Set the text on the "Previous" button.
     * @param text the new text
     * @deprecated use {@link #setControlButtonText(ControlButtonType, String)} with {@link ControlButtonType#PREVIOUS} instead
     */
    @Deprecated
    void setPreviousButtonText(String text);

    /**
     * Set the text on the "Cancel" button.
     * @param text the new text
     * @deprecated use {@link #setControlButtonText(ControlButtonType, String)} with {@link ControlButtonType#CANCEL} instead
     */
    @Deprecated
    void setCancelButtonText(String text);

    /**
     * Install a wizard index on the left side of the window to show overall installation progress.
     * See {@link WizardIndex} for more information. The first step in the wizard index will
     * be highlighted immediately. This setting is not restricted to the current screen, it remains active for subsequent screens.
     * It is suggested to increase the window width of your installer if you use wizard indices.
     * @param wizardIndex the new wizard index or {@code null} if the current wizard index should be removed.
     */
    void setWizardIndex(WizardIndex wizardIndex);

    /**
     * Change the highlighted step in the currently installed wizard index.
     * If no wizard index is set, this has no effect.
     * @param key the index key of the step that should be highlighted. If the key cannot be found
     * in the current wizard index, no step will be highlighted.
     */
    void setWizardIndexKey(String key);

    /**
     * Get the screen style manager. This is useful if you need to nest screen styles or if you have to reference screen styles
     * from screens or form components.
     * @return the screen style manager
     */
    StyleManager getStyleManager();

    /**
     * Sets the size of the application window.
     * This can also be called before any window is shown.
     * @param width the width of the window
     * @param height the height of the window
     */
    void setWindowSize(int width, int height);

    /**
     * Sets the title of the application window.
     * This can also be called before any window is shown.
     * For installer variables in the title, live binding will replace them whenever they change.
     * @param title the title
     */
    void setWindowTitle(String title);
}
