package com.install4j.api.beans;

/**
 * Specifies the error handling for variable replacements. Instances of this class are passed to some of the
 *  {@code replaceVariables(...)} methods in {@link AbstractBean}.
 * <p>
 * Each variable type (installer variable, I18N message, compiler and form variable) has its own error handling.
 * <p>Note: compiler variables in the project are replaced at compile time. Compiler variables in external files,
 * such as a custom localization file, are not replaced at a compile time.
 *
 * @author ej-technologies GmbH
 */
public class VariableErrorHandlingDescriptor {

    /**
     * The default error handling. Throw an exception for undefined installer variables, replace undefined localization keys, and
     * compiler variables with an error message.
     */
    public static final VariableErrorHandlingDescriptor DEFAULT = new VariableErrorHandlingDescriptor(
            VariableErrorHandling.EXCEPTION,
            VariableErrorHandling.ERROR_MESSAGE,
            VariableErrorHandling.ERROR_MESSAGE
        );

    /**
     * Ignore all undefined variables.
     */
    public static final VariableErrorHandlingDescriptor ALWAYS_IGNORE = new VariableErrorHandlingDescriptor(VariableErrorHandling.IGNORE);

    /**
     * Always replace undefined variables with an error message.
     */
    public static final VariableErrorHandlingDescriptor ALWAYS_ERROR_MESSAGE = new VariableErrorHandlingDescriptor(VariableErrorHandling.ERROR_MESSAGE);

    /**
     * Always throw exceptions for undefined variables.
     */
    public static final VariableErrorHandlingDescriptor ALWAYS_EXCEPTION = new VariableErrorHandlingDescriptor(VariableErrorHandling.EXCEPTION);

    private VariableErrorHandling installerVariables;
    private VariableErrorHandling i18nKeys;
    private VariableErrorHandling compilerVariables;
    private VariableErrorHandling formVariables;

    private VariableErrorHandlingDescriptor(VariableErrorHandling variableErrorHandling) {
        this(variableErrorHandling, variableErrorHandling, variableErrorHandling);
    }

    /**
     * Construct a custom variable error handling
     * Like {@link #VariableErrorHandlingDescriptor(VariableErrorHandling, VariableErrorHandling, VariableErrorHandling, VariableErrorHandling)}
     * with formVariables set to {@link VariableErrorHandling#ERROR_MESSAGE}.
     */
    public VariableErrorHandlingDescriptor(VariableErrorHandling installerVariables, VariableErrorHandling i18nKeys, VariableErrorHandling compilerVariables) {
        this(installerVariables, i18nKeys, compilerVariables, VariableErrorHandling.ERROR_MESSAGE);
    }

    /**
     * Construct a custom variable error handling
     * @param installerVariables error handling for installer variables
     * @param i18nKeys error handling for localization keys
     * @param compilerVariables error handling for compiler variables
     * @param formVariables error handling for form variables
     */
    public VariableErrorHandlingDescriptor(VariableErrorHandling installerVariables, VariableErrorHandling i18nKeys, VariableErrorHandling compilerVariables, VariableErrorHandling formVariables) {
        this.installerVariables = installerVariables;
        this.i18nKeys = i18nKeys;
        this.compilerVariables = compilerVariables;
        this.formVariables = formVariables;
    }

    /**
     * Returns the error handling for installer variables.
     * @return the error handling
     */
    public VariableErrorHandling getInstallerVariables() {
        return installerVariables;
    }

    /**
     * Returns the error handling for localization keys.
     * @return the error handling
     */
    public VariableErrorHandling getI18nKeys() {
        return i18nKeys;
    }

    /**
     * Returns the error handling for compiler variables.
     * @return the error handling
     */
    public VariableErrorHandling getCompilerVariables() {
        return compilerVariables;
    }

    /**
     * Returns the error handling for form variables.
     * @return the error handling
     */
    public VariableErrorHandling getFormVariables() {
        return formVariables;
    }
}
